"""
Drag-and-Drop Support for SHAC Studio

Provides OS-level drag-and-drop for audio files with graceful fallback.
Works with tkinterdnd2 if available, provides alternative UI if not.
"""

import logging
import tkinter as tk
from pathlib import Path
from . import theme

logger = logging.getLogger(__name__)

# Try to import tkinterdnd2
try:
    from tkinterdnd2 import DND_FILES, TkinterDnD
    TKDND_AVAILABLE = True
    logger.info("tkinterdnd2 is available - full drag-and-drop enabled")
except ImportError:
    TKDND_AVAILABLE = False
    logger.warning("tkinterdnd2 not available - using fallback drop zone")


class DragDropHandler:
    """
    Handles drag-and-drop operations with visual feedback.

    Works with tkinterdnd2 if available, provides fallback otherwise.
    """

    def __init__(self, widget, callback, file_types=None):
        """
        Initialize drag-drop handler.

        Args:
            widget: The tkinter widget to enable drag-drop on
            callback: Function to call with list of file paths
            file_types: List of allowed extensions (e.g., ['.wav', '.mp3'])
        """
        self.widget = widget
        self.callback = callback
        self.file_types = file_types or ['.wav', '.mp3', '.flac', '.ogg', '.m4a']
        self.is_dragging = False
        self.original_bg = None

        if TKDND_AVAILABLE:
            self._setup_tkdnd()
        else:
            logger.debug("tkinterdnd2 not available - drag-drop disabled")

    def _setup_tkdnd(self):
        """Set up tkinterdnd2 drag-and-drop."""
        try:
            # Register the widget as a drop target
            self.widget.drop_target_register(DND_FILES)

            # Bind drop events
            self.widget.dnd_bind('<<DropEnter>>', self._on_drag_enter)
            self.widget.dnd_bind('<<DropPosition>>', self._on_drag_position)
            self.widget.dnd_bind('<<DropLeave>>', self._on_drag_leave)
            self.widget.dnd_bind('<<Drop>>', self._on_drop)

            logger.info(f"Drag-drop enabled on {self.widget.winfo_class()}")
        except Exception as e:
            logger.error(f"Failed to setup drag-drop: {e}")

    def _on_drag_enter(self, event):
        """Handle drag entering the widget."""
        self.is_dragging = True
        self._highlight_widget(True)
        return event.action

    def _on_drag_position(self, event):
        """Handle drag moving over the widget."""
        return event.action

    def _on_drag_leave(self, event):
        """Handle drag leaving the widget."""
        self.is_dragging = False
        self._highlight_widget(False)
        return event.action

    def _on_drop(self, event):
        """Handle files being dropped."""
        self.is_dragging = False
        self._highlight_widget(False)

        # Parse the file paths from the event data
        files = self._parse_drop_data(event.data)

        # Filter by file type
        valid_files = []
        for filepath in files:
            path = Path(filepath)
            if path.suffix.lower() in self.file_types:
                valid_files.append(str(path))
            else:
                logger.warning(f"Dropped file has unsupported type: {filepath}")

        if valid_files:
            logger.info(f"Dropped {len(valid_files)} valid audio files")
            self.callback(valid_files)
        else:
            logger.warning("No valid audio files in drop")

        return event.action

    def _parse_drop_data(self, data):
        """
        Parse file paths from drop event data.

        tkinterdnd2 returns paths in different formats depending on platform.
        This handles all the edge cases.
        """
        files = []

        # Handle different path formats
        if isinstance(data, str):
            # Could be space-separated or brace-wrapped paths
            if data.startswith('{'):
                # Brace-wrapped paths: {path1} {path2}
                import re
                files = re.findall(r'\{([^}]+)\}', data)
            else:
                # Space-separated paths (may have issues with spaces in names)
                # Try to split intelligently
                parts = data.split()
                current_path = []
                for part in parts:
                    current_path.append(part)
                    # Check if this forms a valid path
                    test_path = ' '.join(current_path)
                    if Path(test_path).exists():
                        files.append(test_path)
                        current_path = []

                # Add any remaining path
                if current_path:
                    files.append(' '.join(current_path))
        elif isinstance(data, (list, tuple)):
            files = list(data)

        return files

    def _highlight_widget(self, highlight):
        """
        Visual feedback during drag operation.

        Args:
            highlight: True to highlight, False to remove highlight
        """
        try:
            if highlight:
                # Store original background
                if self.original_bg is None:
                    self.original_bg = self.widget.cget('background')

                # Highlight with accent color
                self.widget.config(
                    background=theme.ACCENT_PRIMARY,
                    relief='solid',
                    borderwidth=2
                )
            else:
                # Restore original appearance
                if self.original_bg:
                    self.widget.config(
                        background=self.original_bg,
                        relief='flat',
                        borderwidth=0
                    )
                    self.original_bg = None
        except Exception as e:
            logger.debug(f"Could not highlight widget: {e}")


class DropZoneButton(tk.Frame):
    """
    Fallback UI when tkinterdnd2 is not available.

    Provides a visible drop zone that opens a file dialog.
    """

    def __init__(self, parent, callback, file_types=None, **kwargs):
        """
        Create a drop zone button.

        Args:
            parent: Parent widget
            callback: Function to call with list of file paths
            file_types: List of allowed extensions
            **kwargs: Additional frame options
        """
        super().__init__(parent, **kwargs)

        self.callback = callback
        self.file_types = file_types or ['.wav', '.mp3', '.flac', '.ogg', '.m4a']

        # Apply theme
        self.config(
            bg=theme.BACKGROUND_TERTIARY,
            relief='solid',
            borderwidth=2,
            cursor='hand2'
        )

        # Create label
        self.label = tk.Label(
            self,
            text="📁 Click to Add Audio Files\n(Drag-and-drop will be enabled when tkinterdnd2 is installed)",
            font=theme.FONT_NORMAL,
            fg=theme.TEXT_SECONDARY,
            bg=theme.BACKGROUND_TERTIARY,
            cursor='hand2',
            padx=20,
            pady=20
        )
        self.label.pack(fill='both', expand=True)

        # Bind click
        self.bind('<Button-1>', self._on_click)
        self.label.bind('<Button-1>', self._on_click)

        # Hover effects
        self.bind('<Enter>', self._on_enter)
        self.bind('<Leave>', self._on_leave)
        self.label.bind('<Enter>', self._on_enter)
        self.label.bind('<Leave>', self._on_leave)

    def _on_click(self, event):
        """Open file dialog when clicked."""
        from tkinter import filedialog

        # Build file type filters
        filetypes = [
            ("Audio Files", " ".join(f"*{ext}" for ext in self.file_types)),
        ]
        for ext in self.file_types:
            filetypes.append((f"{ext.upper()[1:]} files", f"*{ext}"))
        filetypes.append(("All files", "*.*"))

        files = filedialog.askopenfilenames(
            title="Select Audio Files",
            filetypes=filetypes
        )

        if files:
            self.callback(list(files))

    def _on_enter(self, event):
        """Hover effect."""
        self.config(bg=theme.ACCENT_PRIMARY)
        self.label.config(
            bg=theme.ACCENT_PRIMARY,
            fg='#FFFFFF'
        )

    def _on_leave(self, event):
        """Remove hover effect."""
        self.config(bg=theme.BACKGROUND_TERTIARY)
        self.label.config(
            bg=theme.BACKGROUND_TERTIARY,
            fg=theme.TEXT_SECONDARY
        )


def enable_drop_target(widget, callback, file_types=None):
    """
    Enable drag-and-drop on a widget.

    Args:
        widget: The widget to enable drag-drop on
        callback: Function to call with list of file paths
        file_types: List of allowed extensions (e.g., ['.wav', '.mp3'])

    Returns:
        DragDropHandler instance or None if not supported
    """
    if TKDND_AVAILABLE:
        return DragDropHandler(widget, callback, file_types)
    else:
        logger.info("Drag-drop not available - use DropZoneButton instead")
        return None


def create_drop_zone(parent, callback, file_types=None, **kwargs):
    """
    Create a visible drop zone widget.

    If tkinterdnd2 is available, creates a drop-enabled frame.
    If not, creates a DropZoneButton as fallback.

    Args:
        parent: Parent widget
        callback: Function to call with list of file paths
        file_types: List of allowed extensions
        **kwargs: Additional widget options

    Returns:
        Widget that supports file dropping
    """
    if TKDND_AVAILABLE:
        # Create a themed drop frame
        frame = tk.Frame(
            parent,
            bg=theme.BACKGROUND_TERTIARY,
            relief='solid',
            borderwidth=2,
            **kwargs
        )

        label = tk.Label(
            frame,
            text="📁 Drag Audio Files Here",
            font=theme.FONT_LARGE,
            fg=theme.TEXT_PRIMARY,
            bg=theme.BACKGROUND_TERTIARY,
            padx=20,
            pady=20
        )
        label.pack(fill='both', expand=True)

        # Enable drag-drop
        DragDropHandler(frame, callback, file_types)

        return frame
    else:
        # Use fallback button
        return DropZoneButton(parent, callback, file_types, **kwargs)


# Convenience function for checking availability
def is_available():
    """Check if drag-and-drop is available."""
    return TKDND_AVAILABLE


def get_installation_instructions():
    """Get instructions for installing tkinterdnd2."""
    return """
To enable drag-and-drop functionality, install tkinterdnd2:

pip install tkinterdnd2

Or on Arch Linux:
yay -S python-tkinterdnd2
(or use your preferred AUR helper)

Alternatively, add tkinterdnd2 to requirements.txt and reinstall dependencies.
"""
