"""
SHAC Studio - Professional Icon System
Generates clean, modern icons programmatically for consistent styling
"""

import tkinter as tk
from PIL import Image, ImageDraw, ImageTk
from . import theme


class IconManager:
    """Manages icon creation and caching for SHAC Studio"""

    _cache = {}

    @staticmethod
    def create_icon(icon_name, size=24, color=None):
        """
        Create a professional icon

        Args:
            icon_name: Name of the icon to create
            size: Size in pixels (default 24)
            color: Icon color (default accent primary)

        Returns:
            PhotoImage for use in tkinter
        """
        if color is None:
            color = theme.ACCENT_PRIMARY

        cache_key = f"{icon_name}_{size}_{color}"
        if cache_key in IconManager._cache:
            return IconManager._cache[cache_key]

        # Create icon based on name
        img = Image.new('RGBA', (size, size), (0, 0, 0, 0))
        draw = ImageDraw.Draw(img)

        # Convert hex color to RGB
        r, g, b = IconManager._hex_to_rgb(color)
        fill_color = (r, g, b, 255)

        # Draw the icon
        if icon_name == 'play':
            IconManager._draw_play(draw, size, fill_color)
        elif icon_name == 'pause':
            IconManager._draw_pause(draw, size, fill_color)
        elif icon_name == 'stop':
            IconManager._draw_stop(draw, size, fill_color)
        elif icon_name == 'add':
            IconManager._draw_add(draw, size, fill_color)
        elif icon_name == 'remove':
            IconManager._draw_remove(draw, size, fill_color)
        elif icon_name == 'folder':
            IconManager._draw_folder(draw, size, fill_color)
        elif icon_name == 'save':
            IconManager._draw_save(draw, size, fill_color)
        elif icon_name == 'export':
            IconManager._draw_export(draw, size, fill_color)
        elif icon_name == 'settings':
            IconManager._draw_settings(draw, size, fill_color)
        elif icon_name == 'volume':
            IconManager._draw_volume(draw, size, fill_color)
        elif icon_name == 'mute':
            IconManager._draw_mute(draw, size, fill_color)
        elif icon_name == 'spatial':
            IconManager._draw_spatial(draw, size, fill_color)
        elif icon_name == 'close':
            IconManager._draw_close(draw, size, fill_color)
        else:
            # Default: circle
            IconManager._draw_circle(draw, size, fill_color)

        # Convert to PhotoImage
        photo = ImageTk.PhotoImage(img)
        IconManager._cache[cache_key] = photo
        return photo

    @staticmethod
    def _hex_to_rgb(hex_color):
        """Convert hex color to RGB tuple"""
        hex_color = hex_color.lstrip('#')
        return tuple(int(hex_color[i:i+2], 16) for i in (0, 2, 4))

    # Icon drawing methods
    @staticmethod
    def _draw_play(draw, size, color):
        """Draw play triangle"""
        padding = size * 0.2
        points = [
            (padding, padding),
            (padding, size - padding),
            (size - padding, size / 2)
        ]
        draw.polygon(points, fill=color)

    @staticmethod
    def _draw_pause(draw, size, color):
        """Draw pause bars"""
        padding = size * 0.25
        bar_width = size * 0.15
        gap = size * 0.1

        # Left bar
        draw.rectangle(
            [padding, padding, padding + bar_width, size - padding],
            fill=color
        )
        # Right bar
        draw.rectangle(
            [size - padding - bar_width, padding, size - padding, size - padding],
            fill=color
        )

    @staticmethod
    def _draw_stop(draw, size, color):
        """Draw stop square"""
        padding = size * 0.25
        draw.rectangle(
            [padding, padding, size - padding, size - padding],
            fill=color
        )

    @staticmethod
    def _draw_add(draw, size, color):
        """Draw plus sign"""
        padding = size * 0.2
        thickness = size * 0.15
        center = size / 2

        # Horizontal bar
        draw.rectangle(
            [padding, center - thickness/2, size - padding, center + thickness/2],
            fill=color
        )
        # Vertical bar
        draw.rectangle(
            [center - thickness/2, padding, center + thickness/2, size - padding],
            fill=color
        )

    @staticmethod
    def _draw_remove(draw, size, color):
        """Draw minus sign"""
        padding = size * 0.2
        thickness = size * 0.15
        center = size / 2

        # Horizontal bar only
        draw.rectangle(
            [padding, center - thickness/2, size - padding, center + thickness/2],
            fill=color
        )

    @staticmethod
    def _draw_close(draw, size, color):
        """Draw X for close"""
        padding = size * 0.25
        thickness = 3

        # Draw X with lines
        draw.line(
            [padding, padding, size - padding, size - padding],
            fill=color, width=thickness
        )
        draw.line(
            [size - padding, padding, padding, size - padding],
            fill=color, width=thickness
        )

    @staticmethod
    def _draw_folder(draw, size, color):
        """Draw folder icon"""
        padding = size * 0.2
        tab_width = size * 0.3

        # Folder tab
        draw.rectangle(
            [padding, padding + size * 0.1, padding + tab_width, padding + size * 0.3],
            fill=color
        )
        # Folder body
        draw.rectangle(
            [padding, padding + size * 0.3, size - padding, size - padding],
            fill=color
        )

    @staticmethod
    def _draw_save(draw, size, color):
        """Draw save/floppy disk icon"""
        padding = size * 0.2

        # Outer rectangle
        draw.rectangle(
            [padding, padding, size - padding, size - padding],
            outline=color, width=2
        )
        # Top bar (disk label area)
        draw.rectangle(
            [padding + size * 0.1, padding, size - padding - size * 0.1, padding + size * 0.25],
            fill=color
        )

    @staticmethod
    def _draw_export(draw, size, color):
        """Draw export arrow"""
        padding = size * 0.2
        center = size / 2

        # Arrow shaft
        draw.rectangle(
            [center - size * 0.08, padding + size * 0.3, center + size * 0.08, size - padding],
            fill=color
        )
        # Arrow head
        points = [
            (center, padding),
            (padding, padding + size * 0.3),
            (size - padding, padding + size * 0.3)
        ]
        draw.polygon(points, fill=color)

    @staticmethod
    def _draw_settings(draw, size, color):
        """Draw settings gear icon (simplified)"""
        import math
        center = size / 2
        outer_radius = size * 0.4
        inner_radius = size * 0.2
        teeth = 6

        points = []
        for i in range(teeth * 2):
            angle = (i * math.pi / teeth)
            radius = outer_radius if i % 2 == 0 else inner_radius
            x = center + radius * math.cos(angle)
            y = center + radius * math.sin(angle)
            points.append((x, y))

        draw.polygon(points, fill=color)
        # Center hole
        draw.ellipse(
            [center - size * 0.15, center - size * 0.15,
             center + size * 0.15, center + size * 0.15],
            fill=(0, 0, 0, 0)
        )

    @staticmethod
    def _draw_volume(draw, size, color):
        """Draw volume/speaker icon"""
        padding = size * 0.2

        # Speaker cone
        points = [
            (padding, size * 0.35),
            (padding + size * 0.2, size * 0.35),
            (padding + size * 0.4, padding),
            (padding + size * 0.4, size - padding),
            (padding + size * 0.2, size * 0.65),
            (padding, size * 0.65)
        ]
        draw.polygon(points, fill=color)

        # Sound waves
        for i in range(1, 4):
            radius = padding + size * 0.15 * i
            draw.arc(
                [size * 0.5, size / 2 - radius, size * 0.5 + radius * 2, size / 2 + radius],
                -60, 60, fill=color, width=2
            )

    @staticmethod
    def _draw_mute(draw, size, color):
        """Draw muted volume icon"""
        IconManager._draw_volume(draw, size, color)

        # Add X over it
        padding = size * 0.15
        draw.line(
            [size * 0.5, padding, size - padding, size - padding],
            fill=theme.ERROR, width=3
        )
        draw.line(
            [size - padding, padding, size * 0.5, size - padding],
            fill=theme.ERROR, width=3
        )

    @staticmethod
    def _draw_spatial(draw, size, color):
        """Draw spatial audio icon (concentric circles)"""
        center = size / 2
        for i in range(3):
            radius = size * 0.15 * (i + 1)
            draw.ellipse(
                [center - radius, center - radius, center + radius, center + radius],
                outline=color, width=2
            )
        # Center point
        dot_radius = size * 0.08
        draw.ellipse(
            [center - dot_radius, center - dot_radius,
             center + dot_radius, center + dot_radius],
            fill=color
        )

    @staticmethod
    def _draw_circle(draw, size, color):
        """Draw simple circle (default)"""
        padding = size * 0.2
        draw.ellipse(
            [padding, padding, size - padding, size - padding],
            fill=color
        )


# Convenience function
def get_icon(name, size=24, color=None):
    """Get an icon by name"""
    return IconManager.create_icon(name, size, color)
