"""
State-Aware Revolutionary Spatial Positioning Panel

The core interface for SHAC's breakthrough spatial audio positioning,
now with revolutionary state management for maximum efficiency.
"""

import logging
import tkinter as tk
from tkinter import ttk
from pathlib import Path
import sys
import time
from typing import Dict

sys.path.append(str(Path(__file__).parent.parent))
from ..dockable_panel import DockablePanel
from .. import theme
from ..icons import get_icon
from ..theme_helpers import create_themed_button, create_themed_label, create_themed_frame, configure_ttk_scale_theme

# Import state management
sys.path.insert(0, str(Path(__file__).parent.parent.parent))
from core.state_manager import StatefulComponent, StateEvent, state_manager, state_aware_method

# Set up logger
logger = logging.getLogger(__name__)


class SourcePropertiesPanel(DockablePanel, StatefulComponent):
    """State-aware multi-source spatial positioning interface."""
    
    def __init__(self, spatial_view, parent_window=None):
        DockablePanel.__init__(self, "source_properties", "Spatial Positioning", parent_window)
        StatefulComponent.__init__(self, "SourcePropertiesPanel")
        
        self.spatial_view = spatial_view
        self.current_source_id = None
        self.source_widgets = {}  # Track widgets for each source
        self.audio_engine = None  # Will be set by main window
        
        # State tracking
        self._source_positions = {}  # Cache positions
        self._last_combo_update = 0
        self._preview_timer = None
        # Mute functionality moved to Audio Processing Panel
        
        # Performance tracking
        self._update_count = 0
        self._avoided_updates = 0
        
        # Subscribe to state events
        self._setup_state_subscriptions()
        
        # Default size for this panel
        self.default_width = 350
        self.default_height = 500
        
    def _setup_state_subscriptions(self):
        """Set up state event subscriptions."""
        # Source events
        self.subscribe_to_state(StateEvent.SOURCE_ADDED, 'on_source_added')
        self.subscribe_to_state(StateEvent.SOURCE_REMOVED, 'on_source_removed')
        self.subscribe_to_state(StateEvent.SOURCE_RENAMED, 'on_source_renamed')
        self.subscribe_to_state(StateEvent.SOURCE_POSITION_CHANGED, 'on_source_position_changed')
        self.subscribe_to_state(StateEvent.SELECTION_CHANGED, 'on_selection_changed')
        
        # Mute events
        # Mute event handling moved to Audio Processing Panel
        
        # Project events
        self.subscribe_to_state(StateEvent.PROJECT_LOADED, 'on_project_loaded')
        
        logger.debug(f"{self.component_name} subscribed to state events")
        
    def setup_content(self):
        """Set up the revolutionary spatial positioning interface with professional theme."""
        # Apply theme
        self.content_frame.config(bg=theme.BACKGROUND_SECONDARY)

        # Header with source selection
        header_frame = create_themed_frame(self.content_frame)
        header_frame.pack(fill='x', padx=10, pady=5)

        create_themed_label(header_frame, "Active Source:", style='title').pack(side='left')

        # Themed combobox
        self.source_var = tk.StringVar()
        combo_style = ttk.Style()
        combo_style.configure('Themed.TCombobox',
                            fieldbackground=theme.BACKGROUND_INPUT,
                            background=theme.BACKGROUND_TERTIARY,
                            foreground=theme.TEXT_PRIMARY,
                            borderwidth=0)
        self.source_combo = ttk.Combobox(header_frame, textvariable=self.source_var,
                                        state='readonly', width=20, style='Themed.TCombobox')
        self.source_combo.pack(side='left', padx=(10, 0))
        self.source_combo.bind('<<ComboboxSelected>>', self.on_source_selected)

        # Mute button moved to Audio Processing Panel

        # Create main content with scrolling (themed canvas)
        canvas = tk.Canvas(self.content_frame, highlightthickness=0,
                          bg=theme.BACKGROUND_SECONDARY, bd=0)
        scrollbar = ttk.Scrollbar(self.content_frame, orient="vertical", command=canvas.yview)
        self.scrollable_frame = create_themed_frame(canvas)
        
        self.scrollable_frame.bind(
            "<Configure>",
            lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
        )
        
        canvas.create_window((0, 0), window=self.scrollable_frame, anchor="nw")
        canvas.configure(yscrollcommand=scrollbar.set)
        
        canvas.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="right", fill="y")
        
        # Revolutionary precision controls section
        self.create_precision_controls()
        
        # Quick positioning presets
        self.create_positioning_presets()
        
        # Bulk operations
        self.create_bulk_operations()
        
        # Real-time feedback
        self.create_feedback_display()
        
        # Live preview controls
        # Live preview removed - use Spatial Player for audio feedback
        
        # Performance display
        self.create_performance_display()
        
        # Initial state sync
        self.smart_refresh_source_list()
        
        # Mute button initialization moved to Audio Processing Panel
    
    def create_precision_controls(self):
        """Create the precision positioning controls."""
        precision_frame = ttk.LabelFrame(self.scrollable_frame, text="Precision 3D Positioning")
        precision_frame.pack(fill='x', padx=10, pady=5)
        
        # Add info about fine control
        info_label = ttk.Label(precision_frame, text="Tip: Use mouse wheel for fine adjustments", 
                              foreground='gray', font=('TkDefaultFont', 8))
        info_label.pack(anchor='w', padx=5, pady=(2, 0))
        
        # X Position with enhanced range and precision
        x_frame = ttk.Frame(precision_frame)
        x_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(x_frame, text="X (Left ← → Right):", width=18).pack(side='left')
        
        self.x_var = tk.DoubleVar(value=0.0)
        self.x_scale = ttk.Scale(
            x_frame, 
            from_=-10.0, 
            to=10.0, 
            variable=self.x_var,
            orient='horizontal',
            length=120,
            command=self.on_position_change
        )
        self.x_scale.pack(side='left', padx=5)
        
        # Add fine control with mouse wheel
        self.x_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.x_var, -10, 10))
        self.x_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.x_var, -10, 10))
        self.x_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.x_var, -10, 10))
        
        self.x_entry = ttk.Entry(x_frame, width=8, textvariable=self.x_var)
        self.x_entry.pack(side='left', padx=5)
        self.x_entry.bind('<Return>', self.on_entry_change)
        self.x_entry.bind('<FocusOut>', self.on_entry_change)
        
        # Y Position (Distance)
        y_frame = ttk.Frame(precision_frame)
        y_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(y_frame, text="Y (Back ← → Front):", width=18).pack(side='left')
        
        self.y_var = tk.DoubleVar(value=2.0)
        self.y_scale = ttk.Scale(
            y_frame, 
            from_=-4.0, 
            to=4.0, 
            variable=self.y_var,
            orient='horizontal',
            length=120,
            command=self.on_position_change
        )
        self.y_scale.pack(side='left', padx=5)
        
        # Add fine control with mouse wheel
        self.y_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.y_var, -4, 4))
        self.y_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.y_var, -4, 4))
        self.y_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.y_var, -4, 4))
        
        self.y_entry = ttk.Entry(y_frame, width=8, textvariable=self.y_var)
        self.y_entry.pack(side='left', padx=5)
        self.y_entry.bind('<Return>', self.on_entry_change)
        self.y_entry.bind('<FocusOut>', self.on_entry_change)
        
        # Z Position (Height)
        z_frame = ttk.Frame(precision_frame)
        z_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(z_frame, text="Z (Down ← → Up):", width=18).pack(side='left')
        
        self.z_var = tk.DoubleVar(value=0.0)
        self.z_scale = ttk.Scale(
            z_frame, 
            from_=-10.0, 
            to=10.0, 
            variable=self.z_var,
            orient='horizontal',
            length=120,
            command=self.on_position_change
        )
        self.z_scale.pack(side='left', padx=5)
        
        # Add fine control with mouse wheel
        self.z_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.z_var, -10, 10))
        self.z_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.z_var, -10, 10))
        self.z_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.z_var, -10, 10))
        
        self.z_entry = ttk.Entry(z_frame, width=8, textvariable=self.z_var)
        self.z_entry.pack(side='left', padx=5)
        self.z_entry.bind('<Return>', self.on_entry_change)
        self.z_entry.bind('<FocusOut>', self.on_entry_change)
        
    def create_positioning_presets(self):
        """Create listener spawn position controls."""
        listener_frame = ttk.LabelFrame(self.scrollable_frame, text="Listener Spawn Position")
        listener_frame.pack(fill='x', padx=10, pady=5)
        
        # Create variables for listener position if they don't exist
        if not hasattr(self.spatial_view, 'listener_x'):
            self.spatial_view.listener_x = tk.DoubleVar(value=0.0)
            self.spatial_view.listener_y = tk.DoubleVar(value=0.0)
            self.spatial_view.listener_z = tk.DoubleVar(value=0.0)
        
        # X Position
        x_frame = ttk.Frame(listener_frame)
        x_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(x_frame, text="X:", width=8).pack(side='left')
        
        self.listener_x_scale = ttk.Scale(
            x_frame, 
            from_=-10.0, 
            to=10.0, 
            variable=self.spatial_view.listener_x,
            orient='horizontal',
            length=140,
            command=self.on_listener_position_change
        )
        self.listener_x_scale.pack(side='left', padx=5)
        
        # Add fine control
        self.listener_x_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_x, -10, 10))
        self.listener_x_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_x, -10, 10))
        self.listener_x_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_x, -10, 10))
        
        self.listener_x_entry = ttk.Entry(x_frame, width=8, textvariable=self.spatial_view.listener_x)
        self.listener_x_entry.pack(side='left', padx=5)
        self.listener_x_entry.bind('<Return>', self.on_listener_entry_change)
        self.listener_x_entry.bind('<FocusOut>', self.on_listener_entry_change)
        
        # Y Position
        y_frame = ttk.Frame(listener_frame)
        y_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(y_frame, text="Y:", width=8).pack(side='left')
        
        self.listener_y_scale = ttk.Scale(
            y_frame, 
            from_=-4.0, 
            to=4.0, 
            variable=self.spatial_view.listener_y,
            orient='horizontal',
            length=140,
            command=self.on_listener_position_change
        )
        self.listener_y_scale.pack(side='left', padx=5)
        
        # Add fine control
        self.listener_y_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_y, -4, 4))
        self.listener_y_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_y, -4, 4))
        self.listener_y_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_y, -4, 4))
        
        self.listener_y_entry = ttk.Entry(y_frame, width=8, textvariable=self.spatial_view.listener_y)
        self.listener_y_entry.pack(side='left', padx=5)
        self.listener_y_entry.bind('<Return>', self.on_listener_entry_change)
        self.listener_y_entry.bind('<FocusOut>', self.on_listener_entry_change)
        
        # Z Position
        z_frame = ttk.Frame(listener_frame)
        z_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Label(z_frame, text="Z:", width=8).pack(side='left')
        
        self.listener_z_scale = ttk.Scale(
            z_frame, 
            from_=-10.0, 
            to=10.0, 
            variable=self.spatial_view.listener_z,
            orient='horizontal',
            length=140,
            command=self.on_listener_position_change
        )
        self.listener_z_scale.pack(side='left', padx=5)
        
        # Add fine control
        self.listener_z_scale.bind('<MouseWheel>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_z, -10, 10))
        self.listener_z_scale.bind('<Button-4>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_z, -10, 10))
        self.listener_z_scale.bind('<Button-5>', lambda e: self._fine_adjust_slider(e, self.spatial_view.listener_z, -10, 10))
        
        self.listener_z_entry = ttk.Entry(z_frame, width=8, textvariable=self.spatial_view.listener_z)
        self.listener_z_entry.pack(side='left', padx=5)
        self.listener_z_entry.bind('<Return>', self.on_listener_entry_change)
        self.listener_z_entry.bind('<FocusOut>', self.on_listener_entry_change)
        
    def create_bulk_operations(self):
        """Create bulk operation controls for multiple sources."""
        bulk_frame = ttk.LabelFrame(self.scrollable_frame, text="Source Management")
        bulk_frame.pack(fill='x', padx=10, pady=5)
        
        # Simple separation button
        control_frame = ttk.Frame(bulk_frame)
        control_frame.pack(fill='x', padx=5, pady=5)
        
        ttk.Button(control_frame, text="Separate Sources", width=20,
                  command=self.separate_overlapping_sources).pack(side='left', padx=2)
        
        # Info label
        info_label = ttk.Label(control_frame, text="Untangles overlapping sources", 
                              foreground='gray', font=('TkDefaultFont', 9))
        info_label.pack(side='left', padx=10)
        
    def create_feedback_display(self):
        """Create real-time position feedback display."""
        feedback_frame = ttk.LabelFrame(self.scrollable_frame, text="Real-time Feedback")
        feedback_frame.pack(fill='x', padx=10, pady=5)
        
        # Distance display
        distance_frame = ttk.Frame(feedback_frame)
        distance_frame.pack(fill='x', padx=5, pady=2)
        
        ttk.Label(distance_frame, text="Distance from listener:").pack(side='left')
        self.distance_label = ttk.Label(distance_frame, text="0.0 m", font=('TkFixedFont', 10))
        self.distance_label.pack(side='left', padx=10)
        
        # Angle display
        angle_frame = ttk.Frame(feedback_frame)
        angle_frame.pack(fill='x', padx=5, pady=2)
        
        ttk.Label(angle_frame, text="Angle:").pack(side='left')
        self.angle_label = ttk.Label(angle_frame, text="0°", font=('TkFixedFont', 10))
        self.angle_label.pack(side='left', padx=10)
        
        # Coordinates display
        coord_frame = ttk.Frame(feedback_frame)
        coord_frame.pack(fill='x', padx=5, pady=2)
        
        ttk.Label(coord_frame, text="Coordinates:").pack(side='left')
        self.coord_label = ttk.Label(coord_frame, text="(0.0, 0.0, 0.0)", font=('TkFixedFont', 10))
        self.coord_label.pack(side='left', padx=10)
        
    # Live audio preview removed - use Spatial Player for audio feedback
        
    def create_performance_display(self):
        """Create performance monitoring display."""
        perf_frame = ttk.Frame(self.scrollable_frame)
        perf_frame.pack(fill='x', padx=10, pady=5)
        
        self.perf_label = ttk.Label(perf_frame, text="", 
                                   font=('TkFixedFont', 8), foreground='gray')
        self.perf_label.pack(side='right')
        
        # Start performance monitoring
        self._update_performance_display()
        
    # ========== STATE EVENT HANDLERS ==========
    
    def on_source_added(self, change):
        """Handle source added event."""
        self._log_state_event("Source added", change.source_id)
        self.smart_refresh_source_list()
        
    def on_source_removed(self, change):
        """Handle source removed event."""
        self._log_state_event("Source removed", change.source_id)
        
        # Clear selection if removed source was selected
        if self.current_source_id == change.source_id:
            self.current_source_id = None
            self._clear_position_display()
            
        self.smart_refresh_source_list()
        
    def on_source_renamed(self, change):
        """Handle source renamed event."""
        self._log_state_event("Source renamed", change.source_id)
        self.smart_refresh_source_list()
        
    def on_source_position_changed(self, change):
        """Handle position change from another component."""
        if change.component != self.component_name:
            self._log_state_event("External position change", change.source_id)
            
            # Update cached position
            position = change.data.get('position')
            if position is not None and change.source_id:
                self._source_positions[change.source_id] = position
                
                # If this is the current source, update display
                if change.source_id == self.current_source_id:
                    self._update_position_display(position)
                else:
                    self._log_avoided_update("Position change for non-selected source")
        
    def on_selection_changed(self, change):
        """Handle selection change from another component."""
        if change.component != self.component_name and change.source_id:
            self._log_state_event("External selection change", change.source_id)
            
            # Find and select the source in combo
            if hasattr(self, 'source_combo'):
                source_data = self._get_source_data()
                for i, (sid, name) in enumerate(source_data):
                    if sid == change.source_id:
                        self.source_combo.current(i)
                        self.on_source_selected(None)
                        break
                        
    def on_project_loaded(self, change):
        """Handle project loaded event."""
        self._log_state_event("Project loaded")
        self._source_positions.clear()
        self.smart_refresh_source_list()
        
    @state_aware_method
    def on_source_muted(self, change):
        """Handle source muted event from audio engine."""
        source_id = change.get('source')
        if source_id:
            # Update internal mute state
            self._muted_sources[source_id] = True
            
            # Update button if this is the current source
            if source_id == self.current_source_id:
                self.update_mute_button()
                
            logger.debug(f"Received mute event for {source_id}")
            
    @state_aware_method
    def on_source_unmuted(self, change):
        """Handle source unmuted event from audio engine."""
        source_id = change.get('source')
        if source_id:
            # Update internal mute state
            self._muted_sources[source_id] = False
            
            # Update button if this is the current source
            if source_id == self.current_source_id:
                self.update_mute_button()
                
            logger.debug(f"Received unmute event for {source_id}")
        
    # ========== SMART UPDATE METHODS ==========
    
    def _get_source_data(self):
        """Get source data for combo box - no caching to ensure fresh data."""
        source_data = []

        # Get source names from audio engine
        if self.audio_engine and hasattr(self.audio_engine, 'sources'):
            logger.debug(f"Source Properties: audio_engine.sources has {len(self.audio_engine.sources)} sources")
            for source_id, source in self.audio_engine.sources.items():
                # Try to get name from audio_bridge.loaded_samples first (has friendly names)
                name = source_id  # Default to UUID

                if self.parent_window and hasattr(self.parent_window, 'audio_bridge'):
                    audio_bridge = self.parent_window.audio_bridge
                    if hasattr(audio_bridge, 'loaded_samples') and source_id in audio_bridge.loaded_samples:
                        sample_info = audio_bridge.loaded_samples[source_id]
                        if isinstance(sample_info, dict) and 'name' in sample_info:
                            name = sample_info['name']
                            logger.debug(f"   Got name from audio_bridge: {source_id}: {name}")

                # Fall back to source object's name attribute if audio_bridge didn't work
                if name == source_id and hasattr(source, 'name'):
                    name = getattr(source, 'name', source_id)
                    logger.debug(f"   Got name from source object: {source_id}: {name}")

                source_data.append((source_id, name))
                logger.debug(f"   Final: {source_id}: {name}")
        else:
            logger.warning(f"Source Properties: audio_engine or sources not available")

        return source_data
        
    def smart_refresh_source_list(self):
        """Refresh source list only if changed."""
        current_time = time.time()
        
        try:
            # Get cached or fresh source data
            source_data = self._get_source_data()
            logger.debug(f"Source Properties: Found {len(source_data)} sources to refresh")
            
            # Update combo box values
            if hasattr(self, 'source_combo'):
                current_values = list(self.source_combo['values'])
                new_values = [name for _, name in source_data]
                
                logger.debug(f"Current combo values: {current_values}")
                logger.debug(f"New combo values: {new_values}")
                
                if current_values != new_values:
                    self.source_combo['values'] = new_values
                    
                    # Preserve selection if possible
                    if self.current_source_id:
                        for i, (sid, name) in enumerate(source_data):
                            if sid == self.current_source_id:
                                self.source_combo.current(i)
                                break
                    elif new_values:
                        self.source_combo.current(0)
                        self.on_source_selected(None)
                        
                    self._log_update(f"Source combo refreshed with {len(new_values)} sources")
                    self._last_combo_update = current_time
                    logger.debug(f"Source Properties: Successfully updated combo box")
                else:
                    self._log_avoided_update("Source list unchanged")
            else:
                logger.warning(f"Source Properties: source_combo not found!")
                
        except Exception as e:
            logger.error(f"Error in smart_refresh_source_list: {e}", exc_info=True)
            import traceback
            traceback.print_exc()
            
    def manual_refresh_sources(self):
        """Manually refresh sources while preserving current selection."""
        # Save current selection
        current_selection = self.source_var.get() if hasattr(self, 'source_var') else None
        
        # Call the existing smart refresh
        self.smart_refresh_source_list()
        
        # Try to restore previous selection if it still exists
        if current_selection and hasattr(self, 'source_combo'):
            values = self.source_combo['values']
            if current_selection in values:
                self.source_var.set(current_selection)
                # Trigger selection event to update display
                self.on_source_selected(None)
            else:
                # Previous selection no longer exists
                logger.debug(f"Previous selection '{current_selection}' no longer available")

        logger.debug(f"Manually refreshed source positioning panel")
        
    def toggle_mute(self):
        """Toggle mute state for the currently selected source."""
        if not self.current_source_id:
            return
            
        # Mute toggle moved to Audio Processing Panel
        return
        
        # Mute button update moved to Audio Processing Panel
        
        # Apply mute to audio engine if available
        if self.audio_engine and hasattr(self.audio_engine, 'set_source_mute'):
            self.audio_engine.set_source_mute(self.current_source_id, new_state)
            
        # Emit state change event
        state_manager.emit_event(StateEvent.SOURCE_MUTED if new_state else StateEvent.SOURCE_UNMUTED, {
            'source': self.current_source_id,
            'muted': new_state,
            'component': self.component_name
        })
        
        status = "muted" if new_state else "unmuted"
        logger.debug(f"Source {self.current_source_id} {status}")
        
    def update_mute_button(self):
        """Update mute button appearance based on current source mute state."""
        if not hasattr(self, 'mute_button') or not self.current_source_id:
            return
            
        is_muted = self._muted_sources.get(self.current_source_id, False)
        
        if is_muted:
            self.mute_button.config(text="🔇", style='Toolbutton')  # Muted
        else:
            self.mute_button.config(text="🔊", style='Toolbutton')  # Unmuted
            
        # Enable/disable button based on source availability
        state = 'normal' if self.current_source_id else 'disabled'
        self.mute_button.config(state=state)
        
    def is_source_muted(self, source_id: str) -> bool:
        """Check if a source is currently muted."""
        return self._muted_sources.get(source_id, False)
        
    def get_muted_sources(self) -> Dict[str, bool]:
        """Get all muted sources."""
        return self._muted_sources.copy()
        
    def set_source_mute(self, source_id: str, muted: bool):
        """Set mute state for a source (for external control)."""
        self._muted_sources[source_id] = muted
        
        # Update button if this is the current source
        if source_id == self.current_source_id:
            self.update_mute_button()
            
        # Apply to audio engine if available
        if self.audio_engine and hasattr(self.audio_engine, 'set_source_mute'):
            self.audio_engine.set_source_mute(source_id, muted)
                
    def _update_position_display(self, position):
        """Update position sliders and feedback without triggering events."""
        if position is None:
            return
            
        # Temporarily disable position change callbacks
        self._updating_display = True
        
        try:
            # Update sliders
            self.x_var.set(position[0])
            self.y_var.set(position[1])
            self.z_var.set(position[2])
            
            # Update feedback
            self._update_feedback_display()
            
            self._log_update("Position display updated")
            
        finally:
            self._updating_display = False
            
    def _clear_position_display(self):
        """Clear position display when no source selected."""
        self._updating_display = True
        
        try:
            self.x_var.set(0.0)
            self.y_var.set(0.0)
            self.z_var.set(0.0)
            
            self.distance_label.config(text="--")
            self.angle_label.config(text="--")
            self.coord_label.config(text="--")
            
        finally:
            self._updating_display = False
            
    def _update_feedback_display(self):
        """Update real-time feedback display."""
        if not self.current_source_id:
            return
            
        x = self.x_var.get()
        y = self.y_var.get()
        z = self.z_var.get()
        
        # Calculate distance
        import math
        distance = math.sqrt(x**2 + y**2 + z**2)
        
        # Calculate angle (azimuth)
        angle = math.degrees(math.atan2(x, y))
        
        # Update labels
        self.distance_label.config(text=f"{distance:.1f} m")
        self.angle_label.config(text=f"{angle:.0f}°")
        self.coord_label.config(text=f"({x:.1f}, {y:.1f}, {z:.1f})")
        
    # ========== USER ACTIONS ==========
    
    def on_source_selected(self, event):
        """Handle source selection from combo box."""
        selection = self.source_combo.current()
        if selection >= 0:
            source_data = self._get_source_data()
            if selection < len(source_data):
                source_id, _ = source_data[selection]
                
                if source_id != self.current_source_id:
                    self.current_source_id = source_id
                    
                    # Get position from cache or spatial view
                    position = self._source_positions.get(source_id)
                    if position is None and self.spatial_view:
                        source_info = self.spatial_view.sources.get(source_id, {})
                        position = source_info.get('position', (0, 0, 0))
                        self._source_positions[source_id] = position

                    if position is not None:
                        self._update_position_display(position)
                    
                    # Update mute button for new source
                    self.update_mute_button()
                    
                    # Emit selection event
                    self.emit_state_change(StateEvent.SELECTION_CHANGED, source_id, {
                        'component': 'source_properties_panel'
                    })
                    
                    self._log_update(f"Source selected: {source_id}")
                else:
                    self._log_avoided_update("Same source reselected")
                    
    def on_position_change(self, value=None):
        """Handle position slider change."""
        if hasattr(self, '_updating_display') and self._updating_display:
            return
            
        if not self.current_source_id:
            return
            
        x = self.x_var.get()
        y = self.y_var.get()
        z = self.z_var.get()
        position = (x, y, z)
        
        # Check if position actually changed and save old position for undo
        old_position = self._source_positions.get(self.current_source_id)
        if old_position is not None and abs(old_position[0] - x) < 0.01 and abs(old_position[1] - y) < 0.01 and abs(old_position[2] - z) < 0.01:
            self._log_avoided_update("Position unchanged")
            return

        # Update cache
        self._source_positions[self.current_source_id] = position
        
        # Update spatial view
        if self.spatial_view:
            self.spatial_view.update_source_position(self.current_source_id, position)
        
        # Update main spatial view if available
        if hasattr(self.parent_window, 'spatial_view') and self.parent_window.spatial_view:
            self.parent_window.spatial_view.update_source_position(self.current_source_id, position)
        
        # Update feedback
        self._update_feedback_display()
        
        # Update audio engine
        if self.audio_engine:
            self.audio_engine.set_source_position(self.current_source_id, position)
        
        # Update player visualization
        self.update_player_visualization()

        # Emit state change with old position for undo support
        event_data = {'position': position}
        if old_position is not None:
            event_data['old_position'] = old_position
        self.emit_state_change(StateEvent.SOURCE_POSITION_CHANGED, self.current_source_id, event_data)
        
        self._log_update(f"Position changed: {position}")
        
    def on_entry_change(self, event):
        """Handle manual entry value change."""
        self.on_position_change()
        
    def on_listener_position_change(self, value=None):
        """Handle listener position slider change."""
        try:
            x = self.spatial_view.listener_x.get()
            y = self.spatial_view.listener_y.get()
            z = self.spatial_view.listener_z.get()
            
            # Update spatial view listener position
            if self.spatial_view:
                self.spatial_view.set_listener_position(x, y, z)
                
            # Update audio engine if available
            if self.audio_engine:
                self.audio_engine.set_listener_position(x, y, z)
                
            self._log_update(f"Listener spawn position: ({x:.1f}, {y:.1f}, {z:.1f})")
            
        except (tk.TclError, AttributeError):
            pass  # Invalid input, ignore
            
    def on_listener_entry_change(self, event):
        """Handle manual listener position entry change."""
        self.on_listener_position_change()
        
    def _fine_adjust_slider(self, event, var, min_val, max_val):
        """Fine adjust slider value with mouse wheel for precision control."""
        current = var.get()
        
        # Determine adjustment amount - smaller steps for finer control
        step = 0.1  # 0.1 meter increments
        
        # Mouse wheel up/down or Linux scroll
        if event.delta > 0 or event.num == 4:
            new_val = min(current + step, max_val)
        else:
            new_val = max(current - step, min_val)
            
        var.set(round(new_val, 1))  # Round to 1 decimal place
        
        # Trigger the position change handler
        if var in [self.x_var, self.y_var, self.z_var]:
            self.on_position_change()
        else:
            self.on_listener_position_change()
        
    def separate_overlapping_sources(self):
        """Separate sources that are too close together."""
        if not self.spatial_view:
            return
            
        sources = list(self.spatial_view.sources.keys())
        if len(sources) < 2:
            return  # Nothing to separate
            
        import math
        
        # Get current positions
        positions = {}
        for source_id in sources:
            # spatial_view.sources[source_id] is the position directly (numpy array or tuple)
            pos = self.spatial_view.sources.get(source_id, (0, 0, 0))
            positions[source_id] = list(pos)  # Make mutable
            
        # Define minimum distance between sources (1.5 units seems reasonable)
        min_distance = 1.5
        
        # Simple separation algorithm - move overlapping sources apart
        max_iterations = 10  # Prevent infinite loops
        for iteration in range(max_iterations):
            moved = False
            
            # Check each pair of sources
            for i, source_id1 in enumerate(sources):
                pos1 = positions[source_id1]
                
                for j, source_id2 in enumerate(sources[i+1:], i+1):
                    pos2 = positions[source_id2]
                    
                    # Calculate distance
                    dx = pos2[0] - pos1[0]
                    dy = pos2[1] - pos1[1]
                    dz = pos2[2] - pos1[2]
                    distance = math.sqrt(dx*dx + dy*dy + dz*dz)
                    
                    # If too close, push apart
                    if distance < min_distance and distance > 0.01:  # Avoid division by zero
                        # Calculate push direction
                        push_factor = (min_distance - distance) / 2.0 / distance
                        
                        # Push both sources away from each other
                        positions[source_id1][0] -= dx * push_factor
                        positions[source_id1][1] -= dy * push_factor
                        positions[source_id1][2] -= dz * push_factor
                        
                        positions[source_id2][0] += dx * push_factor
                        positions[source_id2][1] += dy * push_factor
                        positions[source_id2][2] += dz * push_factor
                        
                        moved = True
                    elif distance < 0.01:  # Sources at exact same position
                        # Give them a small random offset
                        import random
                        positions[source_id2][0] += random.uniform(-0.5, 0.5)
                        positions[source_id2][1] += random.uniform(-0.5, 0.5)
                        moved = True
            
            # If nothing moved, we're done
            if not moved:
                break
                
        # Apply the new positions
        sources_moved = 0
        for source_id, new_pos in positions.items():
            old_pos = self.spatial_view.sources.get(source_id, {}).get('position', (0, 0, 0))
            
            # Only update if position actually changed
            if (abs(old_pos[0] - new_pos[0]) > 0.01 or 
                abs(old_pos[1] - new_pos[1]) > 0.01 or 
                abs(old_pos[2] - new_pos[2]) > 0.01):
                
                # Convert back to tuple
                new_pos_tuple = tuple(new_pos)
                
                # Update both spatial view instances
                if self.spatial_view:
                    self.spatial_view.update_source_position(source_id, new_pos_tuple)
                if hasattr(self.parent_window, 'spatial_view') and self.parent_window.spatial_view:
                    self.parent_window.spatial_view.update_source_position(source_id, new_pos_tuple)
                    
                # Cache the position
                self._source_positions[source_id] = new_pos_tuple
                
                # Emit state change
                self.emit_state_change(StateEvent.SOURCE_POSITION_CHANGED, source_id, {
                    'position': new_pos_tuple
                })
                
                sources_moved += 1
                
        # Update visualization
        if sources_moved > 0:
            self._update_all_player_positions()
            self._log_update(f"Separated {sources_moved} overlapping sources")
        else:
            self._log_update("No overlapping sources found")
        
    def toggle_live_preview(self):
        """Toggle live preview mode."""
        if self.preview_enabled_var.get():
            self.enable_preview_mode()
            self._log_update("Live preview enabled")
        else:
            self.disable_preview_mode()
            self._log_update("Live preview disabled")
            
    def enable_preview_mode(self):
        """Enable live preview of position changes."""
        # Implementation depends on audio engine
        pass
        
    def disable_preview_mode(self):
        """Disable live preview mode."""
        # Implementation depends on audio engine
        pass
        
    def play_quick_preview(self):
        """Play a quick 3-second preview."""
        if not self.audio_engine or not self.current_source_id:
            return
            
        # Cancel any existing preview timer
        if self._preview_timer:
            self.content_frame.after_cancel(self._preview_timer)
            
        # Start preview
        self.preview_button.config(text="Playing...", state='disabled')
        
        # Implementation depends on audio engine
        
        # Schedule stop
        self._preview_timer = self.content_frame.after(3000, self._end_quick_preview)
        
    def _end_quick_preview(self):
        """End quick preview."""
        self.preview_button.config(text="Quick Preview (3 seconds)", state='normal')
        self._preview_timer = None
        
    def update_player_visualization(self):
        """Update the spatial player visualization."""
        if hasattr(self.parent_window, 'spatial_player_panel') and self.parent_window.spatial_player_panel:
            player = self.parent_window.spatial_player_panel
            if self.current_source_id:
                position = self._source_positions.get(self.current_source_id)
                if position:
                    player.update_source_position(self.current_source_id, position)
                    player.request_render()  # Force visual update
                    
    def _update_all_player_positions(self):
        """Update all source positions in the player after bulk operations."""
        if hasattr(self.parent_window, 'spatial_player_panel') and self.parent_window.spatial_player_panel:
            player = self.parent_window.spatial_player_panel
            for source_id, position in self._source_positions.items():
                player.update_source_position(source_id, position)
            player.request_render()  # Force visual update
                
    # ========== UTILITY METHODS ==========
    
    def _log_state_event(self, event_type, source_id=None):
        """Log state event reception."""
        msg = f"📡 {event_type}"
        if source_id:
            msg += f" ({source_id})"
        logger.debug(msg)
        
    def _log_update(self, reason):
        """Log a necessary update."""
        self._update_count += 1
        logger.debug(f"Update: {reason}")
        
    def _log_avoided_update(self, reason):
        """Log an avoided unnecessary update."""
        self._avoided_updates += 1
        logger.debug(f"Avoided: {reason}")
        
    def _update_performance_display(self):
        """Update performance display periodically."""
        if hasattr(self, 'perf_label') and self.is_visible:
            total = self._update_count + self._avoided_updates
            if total > 0:
                efficiency = self._avoided_updates / total
                self.perf_label.config(
                    text=f"State efficiency: {efficiency:.0%} updates avoided"
                )
            
            # Schedule next update
            if self.window:
                self.window.after(5000, self._update_performance_display)