/**
 * SHAC Studio Bridge - Real-time communication between Python GUI and Spatial Player
 * 
 * This bridge enables seamless integration of the JavaScript spatial audio engine
 * with the Python-based SHAC Studio interface.
 */

class SHACStudioBridge {
    constructor() {
        this.spatialEngine = null;
        this.isConnected = false;
        this.updateInterval = null;
        
        // Command queue for buffering commands
        this.commandQueue = [];
        this.isProcessing = false;
        
        // Status callbacks
        this.onStatusUpdate = null;
        this.onPositionUpdate = null;
        this.onPlaybackStateChange = null;
        
        // Initialize connection
        this.initialize();
    }
    
    async initialize() {
        console.log('SHAC Studio Bridge initializing...');
        
        // Wait for spatial audio engine to be available
        while (!window.SpatialAudioEngine) {
            await new Promise(resolve => setTimeout(resolve, 100));
        }
        
        // Create spatial audio engine instance
        this.spatialEngine = new window.SpatialAudioEngine();
        await this.spatialEngine.init();
        
        this.isConnected = true;
        console.log('SHAC Studio Bridge connected!');
        
        // Start update loop
        this.startUpdateLoop();
        
        // Process any queued commands
        this.processCommandQueue();
        
        // Notify status
        this.updateStatus('Connected to SHAC Studio');
    }
    
    /**
     * Load spatial audio sources from SHAC data
     */
    async loadSHACData(shacData) {
        if (!this.isConnected) {
            this.queueCommand('loadSHACData', [shacData]);
            return;
        }
        
        try {
            await this.spatialEngine.loadSHAC(shacData);
            this.updateStatus(`Loaded ${Object.keys(shacData.layers).length} spatial sources`);
            console.log('SHAC data loaded into spatial engine');
        } catch (error) {
            console.error('Failed to load SHAC data:', error);
            this.updateStatus('Error loading spatial audio');
        }
    }
    
    /**
     * Transport controls
     */
    async play() {
        if (!this.isConnected) {
            this.queueCommand('play', []);
            return;
        }
        
        await this.spatialEngine.play();
        this.notifyPlaybackState('playing');
    }
    
    async pause() {
        if (!this.isConnected) {
            this.queueCommand('pause', []);
            return;
        }
        
        this.spatialEngine.pause();
        this.notifyPlaybackState('paused');
    }
    
    async stop() {
        if (!this.isConnected) {
            this.queueCommand('stop', []);
            return;
        }
        
        this.spatialEngine.pause();
        // Reset to beginning
        this.spatialEngine.pauseTime = 0;
        this.notifyPlaybackState('stopped');
    }
    
    async replay() {
        if (!this.isConnected) {
            this.queueCommand('replay', []);
            return;
        }
        
        await this.spatialEngine.replay();
        this.notifyPlaybackState('playing');
    }
    
    /**
     * Listener position controls
     */
    updateListenerPosition(x, y, z) {
        if (!this.isConnected) {
            this.queueCommand('updateListenerPosition', [x, y, z]);
            return;
        }
        
        this.spatialEngine.updateListenerPosition(x, y, z);
        console.log(`Listener position: ${x.toFixed(2)}, ${y.toFixed(2)}, ${z.toFixed(2)}`);
    }
    
    updateListenerRotation(azimuth, elevation, roll = 0) {
        if (!this.isConnected) {
            this.queueCommand('updateListenerRotation', [azimuth, elevation, roll]);
            return;
        }
        
        this.spatialEngine.updateListenerRotation(azimuth, elevation, roll);
        console.log(`Listener rotation: ${azimuth.toFixed(1)}°, ${elevation.toFixed(1)}°`);
    }
    
    resetListener() {
        if (!this.isConnected) {
            this.queueCommand('resetListener', []);
            return;
        }
        
        this.spatialEngine.resetListener();
        this.updateStatus('Listener reset to origin');
    }
    
    /**
     * Movement controls
     */
    moveListener(dx, dy, dz) {
        if (!this.isConnected) return;
        this.spatialEngine.moveListener(dx, dy, dz);
    }
    
    moveListenerAbsolute(dx, dy, dz) {
        if (!this.isConnected) return;
        this.spatialEngine.moveListenerAbsolute(dx, dy, dz);
    }
    
    rotateListener(dAzimuth, dElevation) {
        if (!this.isConnected) return;
        this.spatialEngine.rotateListener(dAzimuth, dElevation);
    }
    
    /**
     * Get current state
     */
    getCurrentTime() {
        if (!this.isConnected) return 0;
        return this.spatialEngine.getCurrentTime();
    }
    
    getDuration() {
        if (!this.isConnected) return 0;
        return this.spatialEngine.duration || 0;
    }
    
    getListenerPosition() {
        if (!this.isConnected) return { x: 0, y: 0, z: 0 };
        return this.spatialEngine.listenerPosition;
    }
    
    getListenerRotation() {
        if (!this.isConnected) return { azimuth: 0, elevation: 0, roll: 0 };
        return this.spatialEngine.listenerRotation;
    }
    
    getLayerInfo() {
        if (!this.isConnected) return [];
        return this.spatialEngine.getLayerInfo();
    }
    
    /**
     * Command queue management
     */
    queueCommand(method, args) {
        this.commandQueue.push({ method, args });
        console.log(`Queued command: ${method}`);
    }
    
    async processCommandQueue() {
        if (this.isProcessing || this.commandQueue.length === 0) return;
        
        this.isProcessing = true;
        
        while (this.commandQueue.length > 0) {
            const { method, args } = this.commandQueue.shift();
            
            try {
                await this[method](...args);
            } catch (error) {
                console.error(`Failed to execute queued command ${method}:`, error);
            }
        }
        
        this.isProcessing = false;
    }
    
    /**
     * Update loop for real-time state synchronization
     */
    startUpdateLoop() {
        this.updateInterval = setInterval(() => {
            if (this.isConnected && this.spatialEngine) {
                // Update position if needed
                if (this.onPositionUpdate) {
                    const position = this.getListenerPosition();
                    const rotation = this.getListenerRotation();
                    const currentTime = this.getCurrentTime();
                    
                    this.onPositionUpdate({
                        position,
                        rotation,
                        currentTime,
                        duration: this.getDuration()
                    });
                }
            }
        }, 100); // 10fps updates
    }
    
    stopUpdateLoop() {
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
            this.updateInterval = null;
        }
    }
    
    /**
     * Status and event callbacks
     */
    updateStatus(message) {
        console.log(`SHAC Bridge: ${message}`);
        if (this.onStatusUpdate) {
            this.onStatusUpdate(message);
        }
    }
    
    notifyPlaybackState(state) {
        console.log(`Playback state: ${state}`);
        if (this.onPlaybackStateChange) {
            this.onPlaybackStateChange(state);
        }
    }
    
    /**
     * Cleanup
     */
    disconnect() {
        this.stopUpdateLoop();
        this.isConnected = false;
        
        if (this.spatialEngine) {
            this.spatialEngine.clear();
        }
        
        this.updateStatus('Disconnected from SHAC Studio');
    }
}

// Global bridge instance
window.shacBridge = new SHACStudioBridge();

// Bridge API for external communication
window.SHACStudioAPI = {
    // Transport controls
    play: () => window.shacBridge.play(),
    pause: () => window.shacBridge.pause(),
    stop: () => window.shacBridge.stop(),
    replay: () => window.shacBridge.replay(),
    
    // Position controls
    setListenerPosition: (x, y, z) => window.shacBridge.updateListenerPosition(x, y, z),
    setListenerRotation: (azimuth, elevation, roll) => window.shacBridge.updateListenerRotation(azimuth, elevation, roll),
    moveListener: (dx, dy, dz) => window.shacBridge.moveListener(dx, dy, dz),
    resetListener: () => window.shacBridge.resetListener(),
    
    // Data loading
    loadSHAC: (shacData) => window.shacBridge.loadSHACData(shacData),
    
    // State queries
    getCurrentTime: () => window.shacBridge.getCurrentTime(),
    getDuration: () => window.shacBridge.getDuration(),
    getListenerPosition: () => window.shacBridge.getListenerPosition(),
    getListenerRotation: () => window.shacBridge.getListenerRotation(),
    getLayerInfo: () => window.shacBridge.getLayerInfo(),
    
    // Event callbacks
    onStatusUpdate: (callback) => { window.shacBridge.onStatusUpdate = callback; },
    onPositionUpdate: (callback) => { window.shacBridge.onPositionUpdate = callback; },
    onPlaybackStateChange: (callback) => { window.shacBridge.onPlaybackStateChange = callback; }
};

console.log('SHAC Studio Bridge API loaded');