"""
Dockable Spatial View Panel

The main 3D positioning interface where users can see and manipulate
audio sources in 3D space through drag-and-drop.
"""

import logging
import tkinter as tk
from tkinter import ttk
from pathlib import Path
import sys

# Set up logger
logger = logging.getLogger(__name__)
sys.path.append(str(Path(__file__).parent.parent))
from ..dockable_panel import DockablePanel


class SpatialViewPanel(DockablePanel):
    """Dockable panel for the 3D spatial visualization."""
    
    def __init__(self, spatial_view, parent_window=None):
        super().__init__("spatial_view", "Spatial View", parent_window)
        
        self.spatial_view = spatial_view
        
        # Default size for this panel - larger since it's the main workspace
        self.default_width = 600
        self.default_height = 500
        
    def setup_content(self):
        """Set up the panel content."""
        # Create a frame for the spatial view canvas
        canvas_frame = ttk.Frame(self.content_frame)
        canvas_frame.pack(fill='both', expand=True, padx=5, pady=5)
        
        # Move the spatial view canvas into this panel
        if hasattr(self.spatial_view, 'canvas'):
            # Reparent the canvas to this panel
            self.spatial_view.canvas.pack_forget()
            self.spatial_view.canvas = tk.Canvas(
                canvas_frame,
                bg='#2b2b2b',
                highlightthickness=1,
                highlightbackground='#555'
            )
            self.spatial_view.canvas.pack(fill='both', expand=True)
            
            # Rebind canvas events
            self.spatial_view.canvas.bind('<Configure>', self.spatial_view.on_canvas_resize)
            self.spatial_view.canvas.bind('<Button-1>', self.spatial_view.on_canvas_click)
            self.spatial_view.canvas.bind('<B1-Motion>', self.spatial_view.on_canvas_drag)
            self.spatial_view.canvas.bind('<ButtonRelease-1>', self.spatial_view.on_canvas_release)
            
            # Redraw grid and sources
            self.spatial_view.canvas.after(100, self.spatial_view.draw_grid)
            
        # View controls frame at bottom
        controls_frame = ttk.Frame(self.content_frame)
        controls_frame.pack(fill='x', padx=5, pady=5)
        
        # Grid toggle
        self.grid_var = tk.BooleanVar(value=True)
        grid_check = ttk.Checkbutton(
            controls_frame,
            text="Show Grid",
            variable=self.grid_var,
            command=self.toggle_grid
        )
        grid_check.pack(side='left', padx=5)
        
        # Source labels toggle
        self.labels_var = tk.BooleanVar(value=True)
        labels_check = ttk.Checkbutton(
            controls_frame,
            text="Show Labels", 
            variable=self.labels_var,
            command=self.toggle_labels
        )
        labels_check.pack(side='left', padx=5)
        
        # Elevation lines toggle
        self.elevation_var = tk.BooleanVar(value=True)
        elevation_check = ttk.Checkbutton(
            controls_frame,
            text="Show Elevation",
            variable=self.elevation_var,
            command=self.toggle_elevation
        )
        elevation_check.pack(side='left', padx=5)
        
        
    def toggle_grid(self):
        """Toggle grid visibility."""
        if self.grid_var.get():
            self.spatial_view.draw_grid()
        else:
            self.spatial_view.canvas.delete('grid')
            
    def toggle_labels(self):
        """Toggle source label visibility."""
        # This would require modifying the spatial view to support label visibility
        logger.debug(f"Labels: {'On' if self.labels_var.get() else 'Off'}")
        
    def toggle_elevation(self):
        """Toggle elevation line visibility."""
        # This would require modifying the spatial view to support elevation line visibility
        logger.debug(f"Elevation lines: {'On' if self.elevation_var.get() else 'Off'}")
        
