"""
Dockable Transport Panel

Contains playback controls (play, pause, stop), timeline scrubbing,
and time display. Now fully modular like all other panels.
"""

import logging
import tkinter as tk
from tkinter import ttk
from pathlib import Path
import sys

# Set up logger
logger = logging.getLogger(__name__)
sys.path.append(str(Path(__file__).parent.parent))
from ..dockable_panel import DockablePanel
from .. import theme
from ..icons import get_icon


class TransportPanel(DockablePanel):
    """Dockable panel for transport controls."""
    
    def __init__(self, parent_window=None):
        super().__init__("transport", "Transport", parent_window)
        
        # Transport state
        self.is_playing = False
        self.current_time = 0.0
        self.total_time = 0.0
        
        # Default size for this panel
        self.default_width = 500
        self.default_height = 120
        
        # These will be overridden by main window
        self.play = self.default_play
        self.pause = self.default_pause  
        self.stop = self.default_stop
        self.on_timeline_change = self.default_timeline_change
        
    def setup_content(self):
        """Set up the panel content with professional theme."""
        # Apply theme to content frame
        self.content_frame.config(bg=theme.BACKGROUND_SECONDARY)

        # Controls frame
        controls_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        controls_frame.pack(side='left', padx=10, pady=10)

        # Create icons (white for visibility on colored buttons)
        self.play_icon = get_icon('play', size=20, color='#FFFFFF')
        self.pause_icon = get_icon('pause', size=20, color='#FFFFFF')
        self.stop_icon = get_icon('stop', size=20, color='#FFFFFF')

        # Play button with icon
        self.play_button = tk.Button(
            controls_frame,
            text=" Play",
            image=self.play_icon,
            compound='left',
            command=self.toggle_play,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=12,
            pady=8,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.play_button.pack(side='left', padx=3)

        # Stop button with icon
        self.stop_button = tk.Button(
            controls_frame,
            text=" Stop",
            image=self.stop_icon,
            compound='left',
            command=self.on_stop_click,
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            activeforeground=theme.TEXT_PRIMARY,
            relief='flat',
            borderwidth=0,
            padx=12,
            pady=8,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.stop_button.pack(side='left', padx=3)

        # Add hover effects
        self._add_button_hover_effect(self.play_button)
        self._add_button_hover_effect(self.stop_button)

        # Time display frame
        time_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        time_frame.pack(side='left', padx=20, pady=10)

        # Current time
        self.time_label = tk.Label(
            time_frame,
            text="0:00 / 0:00",
            font=theme.FONT_LARGE,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY
        )
        self.time_label.pack()

        # Timeline frame
        timeline_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        timeline_frame.pack(side='left', fill='x', expand=True, padx=10, pady=10)

        # Progress bar (using ttk Scale with custom styling)
        self.timeline_var = tk.DoubleVar(value=0)

        # Create custom style for the scale
        style = ttk.Style()
        style.configure(
            'Transport.Horizontal.TScale',
            background=theme.BACKGROUND_SECONDARY,
            troughcolor=theme.SLIDER_TRACK,
            borderwidth=0,
            lightcolor=theme.ACCENT_PRIMARY,
            darkcolor=theme.ACCENT_PRIMARY
        )

        self.timeline = ttk.Scale(
            timeline_frame,
            from_=0,
            to=100,
            variable=self.timeline_var,
            orient='horizontal',
            command=self.on_timeline_change,
            style='Transport.Horizontal.TScale'
        )
        self.timeline.pack(fill='x', expand=True)

    def _add_button_hover_effect(self, button):
        """Add hover effect to button"""
        def on_enter(e):
            button.config(bg=theme.BUTTON_HOVER)

        def on_leave(e):
            button.config(bg=theme.BUTTON_NORMAL)

        button.bind('<Enter>', on_enter)
        button.bind('<Leave>', on_leave)
        
    def toggle_play(self):
        """Toggle play/pause."""
        if self.is_playing:
            self.pause()
        else:
            self.play()
            
    def on_stop_click(self):
        """Handle stop button click."""
        self.stop()
        
    def default_play(self):
        """Default play implementation."""
        self.is_playing = True
        self.play_button.config(text=" Pause", image=self.pause_icon)
        self.play_button.config(bg=theme.SUCCESS)  # Green when playing
        logger.debug("Playing...")

    def default_pause(self):
        """Default pause implementation."""
        self.is_playing = False
        self.play_button.config(text=" Play", image=self.play_icon)
        self.play_button.config(bg=theme.BUTTON_NORMAL)
        logger.debug("Paused")

    def default_stop(self):
        """Default stop implementation."""
        self.is_playing = False
        self.play_button.config(text=" Play", image=self.play_icon)
        self.play_button.config(bg=theme.BUTTON_NORMAL)
        self.current_time = 0.0
        self.timeline_var.set(0)
        self.update_time_display()
        logger.debug("Stopped")
        
    def default_timeline_change(self, value):
        """Default timeline change implementation."""
        if self.total_time > 0:
            position = float(value) / 100.0
            self.current_time = position * self.total_time
            self.update_time_display()
            
    def update_time_display(self):
        """Update the time display."""
        current_str = self.format_time(self.current_time)
        total_str = self.format_time(self.total_time)
        self.time_label.config(text=f"{current_str} / {total_str}")
        
    def format_time(self, seconds):
        """Format time as M:SS."""
        minutes = int(seconds // 60)
        secs = int(seconds % 60)
        return f"{minutes}:{secs:02d}"
        
    def set_duration(self, duration):
        """Set total duration."""
        self.total_time = duration
        self.update_time_display()