"""
Project Manager - Save/load functionality

Handles saving and loading SHAC Studio projects, maintaining
source positions and settings between sessions.
"""

import json
import os
import logging
from pathlib import Path
from typing import Dict, List, Any, Optional
from dataclasses import dataclass, asdict

# Set up logger
logger = logging.getLogger(__name__)


@dataclass
class ProjectSource:
    """Project source data."""
    id: str
    name: str
    filepath: str
    position: tuple
    volume: float
    muted: bool


@dataclass 
class ProjectSettings:
    """Project settings data."""
    sample_rate: int
    master_volume: float
    export_order: int
    normalize: bool


class SHACProject:
    """SHAC Studio project data."""
    
    def __init__(self):
        """Initialize empty project."""
        self.name = "Untitled Project"
        self.sources: List[ProjectSource] = []
        self.settings = ProjectSettings(
            sample_rate=48000,
            master_volume=0.8,
            export_order=3,
            normalize=True
        )
        self.filepath: Optional[str] = None
        self.modified = False
        
    def add_source(self, source_id: str, name: str, filepath: str, 
                   position: tuple = (0, 0, 0), volume: float = 1.0, muted: bool = False):
        """Add a source to the project."""
        source = ProjectSource(
            id=source_id,
            name=name,
            filepath=filepath,
            position=position,
            volume=volume,
            muted=muted
        )
        self.sources.append(source)
        self.modified = True
        
    def remove_source(self, source_id: str):
        """Remove a source from the project."""
        self.sources = [s for s in self.sources if s.id != source_id]
        self.modified = True
        
    def update_source(self, source_id: str, **kwargs):
        """Update source properties."""
        for source in self.sources:
            if source.id == source_id:
                for key, value in kwargs.items():
                    if hasattr(source, key):
                        setattr(source, key, value)
                        self.modified = True
                break
                
    def get_source(self, source_id: str) -> Optional[ProjectSource]:
        """Get a source by ID."""
        for source in self.sources:
            if source.id == source_id:
                return source
        return None
        
    def to_dict(self) -> Dict[str, Any]:
        """Convert project to dictionary."""
        return {
            'name': self.name,
            'sources': [asdict(source) for source in self.sources],
            'settings': asdict(self.settings),
            'version': '1.0'
        }
        
    def from_dict(self, data: Dict[str, Any]):
        """Load project from dictionary."""
        self.name = data.get('name', 'Untitled Project')
        
        # Load sources
        self.sources = []
        for source_data in data.get('sources', []):
            source = ProjectSource(**source_data)
            self.sources.append(source)
            
        # Load settings
        settings_data = data.get('settings', {})
        self.settings = ProjectSettings(**settings_data)
        
        self.modified = False


class ProjectManager:
    """Manages SHAC Studio project files."""

    def __init__(self):
        """Initialize project manager."""
        self.current_project = SHACProject()
        self.recent_projects: List[str] = []
        self.max_recent = 10

        # Autosave configuration
        self.autosave_enabled = True
        self.autosave_interval_seconds = 300  # 5 minutes
        self.last_autosave_time = 0

        # Load recent projects list
        self.load_recent_projects()
        
    def new_project(self) -> SHACProject:
        """
        Create a new project.

        Note: Save prompting is handled by the UI layer (ModularMainWindow.new_project)
        before calling this method. This keeps the project manager UI-agnostic.
        """
        self.current_project = SHACProject()
        return self.current_project
        
    def save_project(self, filepath: str = None) -> bool:
        """Save the current project."""
        if filepath is None:
            filepath = self.current_project.filepath
            
        if filepath is None:
            return False
            
        try:
            # Ensure directory exists
            Path(filepath).parent.mkdir(parents=True, exist_ok=True)
            
            # Save project data
            with open(filepath, 'w') as f:
                json.dump(self.current_project.to_dict(), f, indent=2)
                
            self.current_project.filepath = filepath
            self.current_project.modified = False

            # Delete autosave file since we just saved
            self.delete_autosave(filepath)

            # Add to recent projects
            self.add_recent_project(filepath)

            logger.info(f"Project saved: {filepath}")
            return True

        except Exception as e:
            logger.error(f"Error saving project: {e}", exc_info=True)
            return False
            
    def load_project(self, filepath: str) -> bool:
        """Load a project from file."""
        try:
            if not os.path.exists(filepath):
                logger.error(f"Project file not found: {filepath}")
                return False
                
            with open(filepath, 'r') as f:
                data = json.load(f)
                
            # Create new project and load data
            self.current_project = SHACProject()
            self.current_project.from_dict(data)
            self.current_project.filepath = filepath
            
            # Add to recent projects
            self.add_recent_project(filepath)

            logger.info(f"Project loaded: {filepath}")
            return True

        except Exception as e:
            logger.error(f"Error loading project: {e}", exc_info=True)
            return False
            
    def save_as_project(self, filepath: str) -> bool:
        """Save project with new filename."""
        old_filepath = self.current_project.filepath
        self.current_project.filepath = filepath
        
        success = self.save_project(filepath)
        if not success:
            # Restore old filepath if save failed
            self.current_project.filepath = old_filepath
            
        return success
        
    def add_recent_project(self, filepath: str):
        """Add project to recent projects list."""
        # Remove if already in list
        if filepath in self.recent_projects:
            self.recent_projects.remove(filepath)
            
        # Add to beginning
        self.recent_projects.insert(0, filepath)
        
        # Limit to max_recent
        self.recent_projects = self.recent_projects[:self.max_recent]
        
        # Save recent projects
        self.save_recent_projects()
        
    def get_recent_projects(self) -> List[str]:
        """Get list of recent projects."""
        # Filter out non-existent files
        existing = [p for p in self.recent_projects if os.path.exists(p)]
        if len(existing) != len(self.recent_projects):
            self.recent_projects = existing
            self.save_recent_projects()
        return self.recent_projects
        
    def load_recent_projects(self):
        """Load recent projects from config."""
        try:
            config_dir = Path.home() / '.shac_studio'
            config_file = config_dir / 'recent_projects.json'
            
            if config_file.exists():
                with open(config_file, 'r') as f:
                    self.recent_projects = json.load(f)
        except Exception as e:
            logger.warning(f"Error loading recent projects: {e}")
            self.recent_projects = []
            
    def save_recent_projects(self):
        """Save recent projects to config."""
        try:
            config_dir = Path.home() / '.shac_studio'
            config_dir.mkdir(exist_ok=True)
            config_file = config_dir / 'recent_projects.json'
            
            with open(config_file, 'w') as f:
                json.dump(self.recent_projects, f, indent=2)
        except Exception as e:
            logger.warning(f"Error saving recent projects: {e}")
            
    def get_project_name(self) -> str:
        """Get current project name."""
        if self.current_project.filepath:
            return Path(self.current_project.filepath).stem
        return self.current_project.name
        
    def is_project_modified(self) -> bool:
        """Check if current project has unsaved changes."""
        return self.current_project.modified
        
    def mark_modified(self):
        """Mark current project as modified."""
        self.current_project.modified = True

    def get_autosave_path(self) -> Optional[str]:
        """Get the autosave file path for current project."""
        if not self.current_project.filepath:
            # No project file yet - use temp directory
            config_dir = Path.home() / '.shac_studio' / 'autosave'
            config_dir.mkdir(parents=True, exist_ok=True)
            return str(config_dir / 'unsaved_project.shacproject.autosave')
        else:
            # Use same location as project file with .autosave extension
            return str(Path(self.current_project.filepath).with_suffix('.shacproject.autosave'))

    def autosave_project(self) -> bool:
        """Perform an autosave of the current project."""
        import time

        # Check if autosave is enabled and project is modified
        if not self.autosave_enabled:
            return False

        if not self.current_project.modified:
            return False  # No changes to save

        # Check if enough time has passed since last autosave
        current_time = time.time()
        if current_time - self.last_autosave_time < self.autosave_interval_seconds:
            return False  # Too soon

        try:
            autosave_path = self.get_autosave_path()
            if not autosave_path:
                return False

            # Ensure directory exists
            Path(autosave_path).parent.mkdir(parents=True, exist_ok=True)

            # Save project data
            with open(autosave_path, 'w') as f:
                json.dump(self.current_project.to_dict(), f, indent=2)

            self.last_autosave_time = current_time
            logger.info(f"Autosaved project: {autosave_path}")
            return True

        except Exception as e:
            logger.warning(f"Autosave failed: {e}")
            return False

    def check_for_autosave_recovery(self) -> Optional[str]:
        """Check if there's an autosave file to recover."""
        try:
            # Check for unsaved project autosave
            config_dir = Path.home() / '.shac_studio' / 'autosave'
            unsaved_autosave = config_dir / 'unsaved_project.shacproject.autosave'

            if unsaved_autosave.exists():
                # Check if it's recent (less than 24 hours old)
                import time
                age = time.time() - unsaved_autosave.stat().st_mtime
                if age < 86400:  # 24 hours
                    return str(unsaved_autosave)

            return None

        except Exception as e:
            logger.warning(f"Error checking for autosave: {e}")
            return None

    def delete_autosave(self, project_filepath: str = None):
        """Delete autosave file after successful save or explicit project close."""
        try:
            if project_filepath:
                autosave_path = Path(project_filepath).with_suffix('.shacproject.autosave')
            else:
                autosave_path = self.get_autosave_path()

            if autosave_path and Path(autosave_path).exists():
                Path(autosave_path).unlink()
                logger.debug(f"Deleted autosave: {autosave_path}")

        except Exception as e:
            logger.warning(f"Error deleting autosave: {e}")