"""
Dockable Panel System - Modular window architecture for SHAC Studio

Enables Photoshop-style dockable panels that can be shown/hidden,
resized, and magnetically snapped together for a completely
customizable workspace.
"""

import logging
import tkinter as tk
from tkinter import ttk
import json
from pathlib import Path
from typing import Dict, List, Tuple, Optional, Callable

# Set up logger
logger = logging.getLogger(__name__)


class PanelManager:
    """Manages all dockable panels and their interactions."""
    
    def __init__(self, main_window):
        self.main_window = main_window
        self.panels: Dict[str, 'DockablePanel'] = {}
        self.panel_states = {}
        self.snap_distance = 20  # Pixels for magnetic snapping
        self.config_file = Path.home() / ".shac_studio" / "panel_layout.json"
        # Ensure config directory exists
        self.config_file.parent.mkdir(exist_ok=True)
        
        # Load saved panel states
        self.load_panel_states()
        
    def register_panel(self, panel_id: str, panel: 'DockablePanel'):
        """Register a panel with the manager."""
        self.panels[panel_id] = panel
        panel.manager = self
        
        # Apply saved state if available
        if panel_id in self.panel_states:
            state = self.panel_states[panel_id]
            panel.apply_state(state)
            
    def show_panel(self, panel_id: str):
        """Show a panel."""
        if panel_id in self.panels:
            self.panels[panel_id].show()
            
    def hide_panel(self, panel_id: str):
        """Hide a panel."""
        if panel_id in self.panels:
            self.panels[panel_id].hide()
            
    def toggle_panel(self, panel_id: str):
        """Toggle panel visibility."""
        if panel_id in self.panels:
            self.panels[panel_id].toggle()
            
    def is_panel_visible(self, panel_id: str) -> bool:
        """Check if panel is visible."""
        if panel_id in self.panels:
            return self.panels[panel_id].is_visible
        return False
        
    def find_snap_targets(self, moving_panel: 'DockablePanel') -> List[Tuple[str, int, int]]:
        """Find panels that the moving panel can snap to."""
        targets = []
        moving_rect = moving_panel.get_geometry()
        
        for panel_id, panel in self.panels.items():
            if panel != moving_panel and panel.is_visible:
                target_rect = panel.get_geometry()
                
                # Check for potential snaps (within snap_distance)
                if self._can_snap(moving_rect, target_rect):
                    snap_x, snap_y = self._calculate_snap_position(moving_rect, target_rect)
                    targets.append((panel_id, snap_x, snap_y))
                    
        return targets
        
    def _can_snap(self, rect1: Tuple[int, int, int, int], rect2: Tuple[int, int, int, int]) -> bool:
        """Check if two rectangles can snap together."""
        x1, y1, w1, h1 = rect1
        x2, y2, w2, h2 = rect2
        
        # Check if they're close enough horizontally or vertically
        h_close = (abs(x1 - (x2 + w2)) <= self.snap_distance or 
                   abs((x1 + w1) - x2) <= self.snap_distance or
                   abs(x1 - x2) <= self.snap_distance)
                   
        v_close = (abs(y1 - (y2 + h2)) <= self.snap_distance or
                   abs((y1 + h1) - y2) <= self.snap_distance or
                   abs(y1 - y2) <= self.snap_distance)
                   
        # They can snap if they're close in one dimension and overlapping in the other
        h_overlap = not (x1 + w1 < x2 or x2 + w2 < x1)
        v_overlap = not (y1 + h1 < y2 or y2 + h2 < y1)
        
        return (h_close and v_overlap) or (v_close and h_overlap)
        
    def _calculate_snap_position(self, moving_rect: Tuple[int, int, int, int], 
                                target_rect: Tuple[int, int, int, int]) -> Tuple[int, int]:
        """Calculate the snap position for magnetic docking."""
        x1, y1, w1, h1 = moving_rect
        x2, y2, w2, h2 = target_rect
        
        # Try different snap positions and return the closest
        snap_positions = [
            (x2 + w2, y1),  # Right of target
            (x2 - w1, y1),  # Left of target
            (x1, y2 + h2),  # Below target
            (x1, y2 - h1),  # Above target
            (x2, y1),       # Same X as target
            (x1, y2),       # Same Y as target
        ]
        
        # Find closest valid snap position
        current_pos = (x1, y1)
        closest_snap = min(snap_positions, 
                          key=lambda pos: abs(pos[0] - current_pos[0]) + abs(pos[1] - current_pos[1]))
        
        return closest_snap
        
    def save_panel_states(self):
        """Save current panel states to file."""
        states = {}
        for panel_id, panel in self.panels.items():
            states[panel_id] = panel.get_state()
            
        try:
            with open(self.config_file, 'w') as f:
                json.dump(states, f, indent=2)
        except Exception as e:
            logger.warning(f"Could not save panel states: {e}")
            
    def load_panel_states(self):
        """Load panel states from file."""
        try:
            if self.config_file.exists():
                with open(self.config_file, 'r') as f:
                    self.panel_states = json.load(f)
        except Exception as e:
            logger.warning(f"Could not load panel states: {e}")
            self.panel_states = {}


class DockablePanel:
    """Base class for dockable panels in SHAC Studio."""

    def __init__(self, panel_id: str, title: str, parent_window=None, enable_transitions=True):
        self.panel_id = panel_id
        self.title = title
        self.parent_window = parent_window
        self.manager: Optional[PanelManager] = None

        # Window state
        self.window: Optional[tk.Toplevel] = None
        self.is_visible = False
        self.is_docked = False

        # Default geometry
        self.default_width = 300
        self.default_height = 400
        self.default_x = 100
        self.default_y = 100

        # Content frame (to be filled by subclasses)
        self.content_frame: Optional[ttk.Frame] = None

        # Transition settings
        self.enable_transitions = enable_transitions
        self._fade_steps = 8  # Number of steps in fade animation
        self._fade_duration = 180  # Total duration in ms (subtle and quick)
        self._animating = False
        
    def create_window(self):
        """Create the panel window using OS title bar for dragging."""
        if self.window is None:
            # Use tk_parent if available (for cases where parent_window is not a Tk widget)
            tk_parent = getattr(self, 'tk_parent', self.parent_window)
            self.window = tk.Toplevel(tk_parent)
            self.window.title(self.title)
            self.window.geometry(f"{self.default_width}x{self.default_height}+{self.default_x}+{self.default_y}")

            # Set panel icon (same as main window)
            try:
                icon_path = Path(__file__).parent.parent / "assets" / "shac-icon.png"
                if icon_path.exists():
                    icon = tk.PhotoImage(file=str(icon_path))
                    self.window.iconphoto(True, icon)
            except Exception:
                pass  # Silently fail - icon is non-critical

            # Create content frame directly - OS handles dragging via title bar
            # Using tk.Frame (not ttk.Frame) so panels can set bg color for theming
            # Import theme here to avoid circular imports
            from . import theme
            self.content_frame = tk.Frame(self.window, bg=theme.BACKGROUND_SECONDARY)
            self.content_frame.pack(fill='both', expand=True, padx=5, pady=5)

            # Theme the window itself
            self.window.config(bg=theme.BACKGROUND_SECONDARY)
            
            # Set up window events (NO custom drag events - OS handles it)
            self.window.protocol("WM_DELETE_WINDOW", self.hide)
            self.window.bind("<Configure>", self.on_configure)
            
            # Initially hidden
            self.window.withdraw()
            
            # Let subclass set up content
            self.setup_content()
            
    def setup_content(self):
        """Override this method to set up panel content."""
        pass
        
    def show(self):
        """Show the panel with subtle fade-in animation."""
        if self.window is None:
            self.create_window()

        # If already visible, just lift to front
        if self.is_visible:
            self.window.lift()
            return

        self.is_visible = True

        # Show the window immediately but potentially transparent
        if self.enable_transitions and not self._animating:
            # Start with window transparent for fade-in
            try:
                self.window.attributes('-alpha', 0.0)
                self.window.deiconify()
                self.window.lift()
                self._animate_fade_in()
            except tk.TclError:
                # Some window managers don't support alpha, fallback to instant
                self.window.deiconify()
                self.window.lift()
        else:
            # Instant show (no transition)
            self.window.deiconify()
            self.window.lift()
        
    def hide(self):
        """Hide the panel."""
        if self.window:
            self.window.withdraw()
        self.is_visible = False
        
        # Save state when hiding
        if self.manager:
            self.manager.save_panel_states()
            
    def toggle(self):
        """Toggle panel visibility."""
        if self.is_visible:
            self.hide()
        else:
            self.show()
            
    def get_geometry(self) -> Tuple[int, int, int, int]:
        """Get panel geometry as (x, y, width, height)."""
        if self.window and self.is_visible:
            self.window.update_idletasks()
            x = self.window.winfo_x()
            y = self.window.winfo_y()
            width = self.window.winfo_width()
            height = self.window.winfo_height()
            return (x, y, width, height)
        return (0, 0, 0, 0)
        
    def set_geometry(self, x: int, y: int, width: int, height: int):
        """Set panel geometry."""
        if self.window:
            self.window.geometry(f"{width}x{height}+{x}+{y}")
            
    def get_state(self) -> Dict:
        """Get current panel state for saving."""
        if self.window and self.is_visible:
            x, y, width, height = self.get_geometry()
            return {
                'visible': self.is_visible,
                'x': x,
                'y': y,
                'width': width,
                'height': height
            }
        return {
            'visible': self.is_visible,
            'x': self.default_x,
            'y': self.default_y,
            'width': self.default_width,
            'height': self.default_height
        }
        
    def apply_state(self, state: Dict):
        """Apply saved state to panel."""
        if 'visible' in state and state['visible']:
            self.show()
            if self.window:
                x = state.get('x', self.default_x)
                y = state.get('y', self.default_y)
                width = state.get('width', self.default_width)
                height = state.get('height', self.default_height)
                self.set_geometry(x, y, width, height)
        else:
            self.hide()
            
    def on_configure(self, event):
        """Handle window configuration changes."""
        # Save state on resize/move (OS handles dragging automatically)
        if self.manager and event.widget == self.window:
            self.manager.save_panel_states()

    def _animate_fade_in(self, step=0):
        """Animate a subtle fade-in effect for the panel."""
        if not self.window or not self.is_visible:
            self._animating = False
            return

        self._animating = True

        # Calculate alpha for this step (ease-out curve for smoothness)
        progress = step / self._fade_steps
        # Ease-out cubic for smooth deceleration
        alpha = 1 - pow(1 - progress, 3)

        try:
            self.window.attributes('-alpha', alpha)
        except tk.TclError:
            # Window manager doesn't support alpha, stop animation
            self._animating = False
            return

        if step < self._fade_steps:
            # Schedule next step
            delay = self._fade_duration // self._fade_steps
            self.window.after(delay, lambda: self._animate_fade_in(step + 1))
        else:
            # Animation complete
            self._animating = False
            # Ensure fully opaque
            try:
                self.window.attributes('-alpha', 1.0)
            except tk.TclError:
                pass

    def destroy(self):
        """Destroy the panel and clean up resources."""
        # Stop any ongoing animations
        self._animating = False

        # Call cleanup if this panel is a StatefulComponent
        if hasattr(self, 'cleanup_state_subscriptions'):
            self.cleanup_state_subscriptions()

        # Destroy the window
        if self.window:
            self.window.destroy()
            self.window = None

        self.is_visible = False