"""
State-Aware Audio Sources Panel

This panel uses the revolutionary state management system to eliminate
unnecessary updates and provide efficient source management.
"""

import logging
import tkinter as tk
from tkinter import ttk, filedialog, simpledialog, messagebox
from pathlib import Path
import sys
import time

# Set up logger
logger = logging.getLogger(__name__)

sys.path.append(str(Path(__file__).parent.parent))
from ..dockable_panel import DockablePanel
from .. import theme
from ..icons import get_icon
from .. import error_dialogs
from .. import drag_drop

# Import state management
sys.path.insert(0, str(Path(__file__).parent.parent.parent))
from core.state_manager import StatefulComponent, StateEvent, state_manager, state_aware_method


class AudioSourcesPanel(DockablePanel, StatefulComponent):
    """State-aware audio sources panel that only updates when necessary."""
    
    def __init__(self, audio_bridge, parent_window=None):
        DockablePanel.__init__(self, "audio_sources", "Audio Sources", parent_window)
        StatefulComponent.__init__(self, "AudioSourcesPanel")
        
        self.audio_bridge = audio_bridge
        self.sources = []  # List of loaded audio sources
        self.spatial_view = None  # Will be set by main window
        
        # Performance tracking
        self._update_count = 0
        self._avoided_updates = 0
        self._last_tree_update = 0
        
        # Subscribe to relevant state events
        self._setup_state_subscriptions()
        
        # Default size for this panel
        self.default_width = 350
        self.default_height = 400
        
    def _setup_state_subscriptions(self):
        """Set up state event subscriptions."""
        # Source management events
        self.subscribe_to_state(StateEvent.SOURCE_ADDED, 'on_source_added')
        self.subscribe_to_state(StateEvent.SOURCE_REMOVED, 'on_source_removed')
        self.subscribe_to_state(StateEvent.SOURCE_RENAMED, 'on_source_renamed')
        self.subscribe_to_state(StateEvent.SOURCE_MODIFIED, 'on_source_modified')
        self.subscribe_to_state(StateEvent.SOURCE_POSITION_CHANGED, 'on_source_position_changed')
        
        # Project events
        self.subscribe_to_state(StateEvent.PROJECT_LOADED, 'on_project_loaded')
        self.subscribe_to_state(StateEvent.PROJECT_CLOSED, 'on_project_closed')
        
        logger.debug(f"{self.component_name} subscribed to state events")
        
    def set_spatial_view(self, spatial_view):
        """Set reference to spatial view."""
        self.spatial_view = spatial_view
        if spatial_view:
            spatial_view.source_panel = self  # Set back-reference
        
    def setup_content(self):
        """Set up the panel content with professional theme."""
        # Apply theme to content frame
        self.content_frame.config(bg=theme.BACKGROUND_SECONDARY)

        # Create icons
        self.add_icon = get_icon('add', size=18, color='#FFFFFF')
        self.folder_icon = get_icon('folder', size=18, color='#FFFFFF')

        # Button frame for add and refresh
        button_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        button_frame.pack(fill='x', pady=10, padx=10)

        # Add source button with icon
        self.add_button = tk.Button(
            button_frame,
            text=" Add Audio Files",
            image=self.folder_icon,
            compound='left',
            command=self.add_sources,
            bg=theme.ACCENT_PRIMARY,
            fg='#FFFFFF',
            activebackground=theme.ACCENT_HOVER,
            relief='flat',
            borderwidth=0,
            padx=12,
            pady=8,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.add_button.pack(side='left', fill='x', expand=True)

        # Add hover effect
        def on_enter(e):
            self.add_button.config(bg=theme.ACCENT_HOVER)
        def on_leave(e):
            self.add_button.config(bg=theme.ACCENT_PRIMARY)
        self.add_button.bind('<Enter>', on_enter)
        self.add_button.bind('<Leave>', on_leave)

        # Drag-and-drop zone
        if drag_drop.is_available():
            # Full drag-and-drop with tkinterdnd2
            logger.info("Enabling drag-and-drop on Audio Sources panel")
            drag_drop.enable_drop_target(
                self.content_frame,
                self._on_files_dropped,
                file_types=['.wav', '.mp3', '.flac', '.ogg', '.m4a']
            )

            # Add subtle hint
            drop_hint = tk.Label(
                button_frame,
                text="  (or drag files here)",
                font=theme.FONT_SMALL,
                fg=theme.TEXT_SECONDARY,
                bg=theme.BACKGROUND_SECONDARY
            )
            drop_hint.pack(side='left', padx=5)
        else:
            # Show info about installing tkinterdnd2
            info_label = tk.Label(
                button_frame,
                text="  (drag-and-drop available with tkinterdnd2)",
                font=theme.FONT_SMALL,
                fg=theme.TEXT_SECONDARY,
                bg=theme.BACKGROUND_SECONDARY
            )
            info_label.pack(side='left', padx=5)

        # Performance info (subtle)
        perf_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        perf_frame.pack(fill='x', padx=10)

        self.perf_label = tk.Label(
            perf_frame,
            text="",
            font=theme.FONT_SMALL,
            fg=theme.TEXT_SECONDARY,
            bg=theme.BACKGROUND_SECONDARY
        )
        self.perf_label.pack(side='right')

        # Source list frame
        list_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        list_frame.pack(fill='both', expand=True, padx=10, pady=5)

        # Scrollbar with theme
        scrollbar = ttk.Scrollbar(list_frame)
        scrollbar.pack(side='right', fill='y')

        # Treeview for source list with softer mid-gray background
        style = ttk.Style()
        style.configure(
            'Sources.Treeview',
            background=theme.BACKGROUND_LISTBOX,  # Softer mid-gray
            fieldbackground=theme.BACKGROUND_LISTBOX,
            foreground=theme.TEXT_PRIMARY,
            borderwidth=0,
            relief='flat'
        )
        style.configure(
            'Sources.Treeview.Heading',
            background=theme.BACKGROUND_TERTIARY,
            foreground=theme.TEXT_PRIMARY,
            borderwidth=0,
            relief='flat'
        )
        style.map('Sources.Treeview',
                  background=[('selected', theme.ACCENT_PRIMARY)],
                  foreground=[('selected', '#FFFFFF')])

        self.source_tree = ttk.Treeview(
            list_frame,
            yscrollcommand=scrollbar.set,
            columns=('duration', 'position'),
            show='tree headings',
            style='Sources.Treeview'
        )
        self.source_tree.pack(side='left', fill='both', expand=True)
        scrollbar.config(command=self.source_tree.yview)

        # Configure columns
        self.source_tree.heading('#0', text='Name')
        self.source_tree.heading('duration', text='Duration')
        self.source_tree.heading('position', text='Position')

        self.source_tree.column('#0', width=180, minwidth=100)
        self.source_tree.column('duration', width=70, minwidth=60)
        self.source_tree.column('position', width=80, minwidth=70)
        
        # Controls frame
        controls_frame = tk.Frame(self.content_frame, bg=theme.BACKGROUND_SECONDARY)
        controls_frame.pack(fill='x', padx=10, pady=5)

        # Create icons for control buttons
        self.remove_icon = get_icon('remove', size=16, color=theme.TEXT_PRIMARY)
        self.close_icon = get_icon('close', size=16, color=theme.TEXT_PRIMARY)

        # Remove button
        self.remove_button = tk.Button(
            controls_frame,
            text=" Remove",
            image=self.remove_icon,
            compound='left',
            command=self.remove_source,
            state='disabled',
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            disabledforeground=theme.TEXT_DISABLED,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.remove_button.pack(side='left', padx=2)

        # Clear all button
        self.clear_button = tk.Button(
            controls_frame,
            text=" Clear All",
            image=self.close_icon,
            compound='left',
            command=self.clear_sources,
            state='disabled',
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            disabledforeground=theme.TEXT_DISABLED,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.clear_button.pack(side='left', padx=2)

        # Rename button
        self.rename_button = tk.Button(
            controls_frame,
            text="Rename",
            command=self.rename_selected_source,
            state='disabled',
            bg=theme.BUTTON_NORMAL,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.BUTTON_HOVER,
            disabledforeground=theme.TEXT_DISABLED,
            relief='flat',
            borderwidth=0,
            padx=10,
            pady=6,
            font=theme.FONT_NORMAL,
            cursor='hand2'
        )
        self.rename_button.pack(side='left', padx=2)
        
        # Mute/Solo controls moved to Source Control Panel
        
        # Track currently selected source
        self.selected_source_id = None
        
        # Track mute/solo states for sources
        self.muted_sources = set()  # Set of muted source IDs
        self.soloed_sources = set()  # Set of soloed source IDs
        
        # Bind selection event
        self.source_tree.bind('<<TreeviewSelect>>', self.on_select)

        # Bind double-click for rename
        self.source_tree.bind('<Double-1>', self.on_double_click)

        # Create right-click context menu
        self._create_context_menu()

        # Bind right-click to show context menu
        self.source_tree.bind('<Button-3>', self.show_context_menu)  # Right-click on most systems
        self.source_tree.bind('<Button-2>', self.show_context_menu)  # macOS right-click (sometimes)

        # Initial button state update
        self._update_button_states()
        
        # Track mute/solo states
        self.muted_sources = set()  # Set of muted source IDs
        self.soloed_sources = set()  # Set of soloed source IDs
        
        # Update performance display periodically
        self._update_performance_display()

        # If sources already exist (e.g., from project loading), refresh the tree
        logger.info("Panel setup_content complete - checking for existing sources...")
        if self.audio_bridge and hasattr(self.audio_bridge, 'main_window'):
            main_window = self.audio_bridge.main_window
            if main_window and hasattr(main_window, 'audio_engine'):
                audio_engine = main_window.audio_engine
                if audio_engine.sources:
                    logger.info(f"Panel initialized with {len(audio_engine.sources)} existing sources - refreshing")
                    self.smart_refresh_tree()
                else:
                    logger.info("No sources found in audio engine yet")
            else:
                logger.info("Main window or audio engine not available yet")
        else:
            logger.info("audio_bridge or main_window not available yet")

    def _create_context_menu(self):
        """Create right-click context menu for source list."""
        self.context_menu = tk.Menu(
            self.source_tree,
            tearoff=0,
            bg=theme.BACKGROUND_SECONDARY,
            fg=theme.TEXT_PRIMARY,
            activebackground=theme.ACCENT_PRIMARY,
            activeforeground='#FFFFFF',
            borderwidth=0
        )

        # Add menu items
        self.context_menu.add_command(
            label="Rename",
            command=self.rename_selected_source
        )
        self.context_menu.add_command(
            label="Remove",
            command=self.remove_source
        )
        self.context_menu.add_separator()
        self.context_menu.add_command(
            label="Clear All Sources",
            command=self.clear_sources
        )

    def show_context_menu(self, event):
        """Show context menu at cursor position."""
        # Select the item under cursor
        item = self.source_tree.identify_row(event.y)
        if item:
            self.source_tree.selection_set(item)
            # Show menu at cursor position
            try:
                self.context_menu.tk_popup(event.x_root, event.y_root)
            finally:
                self.context_menu.grab_release()

    # ========== STATE EVENT HANDLERS ==========
    
    def on_source_added(self, change):
        """Handle source added by another component."""
        # If we added it ourselves, we already updated the UI
        if change.component != self.component_name:
            self._log_state_event("External source added", change.source_id)
            self.smart_refresh_tree()
            
    def on_source_removed(self, change):
        """Handle source removed by another component."""
        if change.component != self.component_name:
            self._log_state_event("External source removed", change.source_id)
            self.smart_refresh_tree()
            
    def on_source_renamed(self, change):
        """Handle source renamed."""
        self._log_state_event("Source renamed", change.source_id)
        # Only update the specific source in the tree
        self._update_single_source_in_tree(change.source_id)
        
    def on_source_modified(self, change):
        """Handle source modified."""
        self._log_state_event("Source modified", change.source_id)
        # Only update if it affects display (duration change, etc)
        if change.data.get('affects_display', True):
            self._update_single_source_in_tree(change.source_id)
        else:
            self._log_avoided_update("Modification doesn't affect display")
            
    def on_source_position_changed(self, change):
        """Handle source position change."""
        # Only update the position column for that specific source
        self._update_source_position_in_tree(change.source_id, change.data.get('position'))
        
    def on_project_loaded(self, change):
        """Handle project loaded - full refresh needed."""
        self._log_state_event("Project loaded")
        self.smart_refresh_tree()
        
    def on_project_closed(self, change):
        """Handle project closed - clear everything."""
        self._log_state_event("Project closed")
        self.clear_sources()
        
    # ========== SMART UPDATE METHODS ==========
    
    @state_aware_method(cache_key="source_list", max_age=2.0)
    def get_source_list_data(self):
        """Get source list data with caching."""
        source_data = []

        # Get audio engine through main window
        audio_engine = None
        if self.audio_bridge and hasattr(self.audio_bridge, 'main_window'):
            main_window = self.audio_bridge.main_window
            if main_window and hasattr(main_window, 'audio_engine'):
                audio_engine = main_window.audio_engine

        if hasattr(self.audio_bridge, 'loaded_samples'):
            for source_id, sample_info in self.audio_bridge.loaded_samples.items():
                # Get position from audio engine if available
                position = (0, 0, 0)
                if audio_engine and source_id in audio_engine.sources:
                    position = audio_engine.sources[source_id].position

                # Get name from our internal list if available, otherwise from sample info
                name = sample_info.get('name', 'Unknown')
                for source in self.sources:
                    if source['id'] == source_id:
                        name = source['name']
                        break

                source_data.append({
                    'id': source_id,
                    'name': name,
                    'duration': sample_info.get('duration', 0),
                    'position': position
                })

        return source_data
    
    def smart_refresh_tree(self):
        """Refresh tree only if source list actually changed."""
        current_time = time.time()
        
        # Get current tree state
        current_tree_ids = set(self.source_tree.get_children())
        current_source_ids = {s['id'] for s in self.sources}
        
        # Get actual source state
        actual_source_ids = set(self.audio_bridge.loaded_samples.keys()) if hasattr(self.audio_bridge, 'loaded_samples') else set()
        
        # Check if anything changed
        if current_source_ids == actual_source_ids and current_time - self._last_tree_update < 1.0:
            self._log_avoided_update("Source list unchanged")
            return
        
        # Something changed - update tree
        self._perform_tree_update()
        self._last_tree_update = current_time
        
    def _perform_tree_update(self):
        """Actually update the tree."""
        # Clear and rebuild (could be optimized further)
        self.source_tree.delete(*self.source_tree.get_children())

        # Get source data (which now works even if self.sources is empty)
        source_data = self.get_source_list_data()

        # Sync self.sources with actual source data and populate tree
        self.sources = []
        for source in source_data:
            duration_str = self.format_duration(source['duration'])
            position = source.get('position', (0, 0, 0))
            position_str = f"{position[0]:.1f}, {position[1]:.1f}, {position[2]:.1f}"

            item_id = self.source_tree.insert(
                '',
                'end',
                text=source['name'],
                values=(duration_str, position_str)
            )

            # Add to self.sources with tree_id
            self.sources.append({
                'id': source['id'],
                'name': source['name'],
                'duration': source['duration'],
                'position': position,
                'tree_id': item_id
            })

        self._log_update("Tree fully refreshed")
        self._update_button_states()
        
    def _update_single_source_in_tree(self, source_id):
        """Update just one source in the tree."""
        for source in self.sources:
            if source['id'] == source_id:
                if 'tree_id' in source and self.source_tree.exists(source['tree_id']):
                    duration_str = self.format_duration(source['duration'])
                    position = source.get('position', (0, 0, 0))
                    position_str = f"{position[0]:.1f}, {position[1]:.1f}, {position[2]:.1f}"
                    
                    self.source_tree.item(source['tree_id'], 
                                        text=source['name'],
                                        values=(duration_str, position_str))
                    self._log_update(f"Updated source {source_id}")
                else:
                    # Source exists but not in tree - need full refresh
                    self.smart_refresh_tree()
                return
                
    def _update_source_position_in_tree(self, source_id, position):
        """Update just the position column for a source."""
        if not position:
            return
            
        for source in self.sources:
            if source['id'] == source_id:
                source['position'] = position
                if 'tree_id' in source and self.source_tree.exists(source['tree_id']):
                    # Get current values
                    current_values = self.source_tree.item(source['tree_id'], 'values')
                    if current_values:
                        duration_str = current_values[0]  # Keep existing duration
                        position_str = f"{position[0]:.1f}, {position[1]:.1f}, {position[2]:.1f}"
                        
                        # Only update if position actually changed
                        if current_values[1] != position_str:
                            self.source_tree.item(source['tree_id'], 
                                                values=(duration_str, position_str))
                            self._log_update(f"Position updated for {source_id}")
                        else:
                            self._log_avoided_update("Position unchanged")
                return
        
    def _update_button_states(self):
        """Update button states based on current selection and sources."""
        if self.sources:
            self.clear_button.config(state='normal')
        else:
            self.clear_button.config(state='disabled')
            
        if self.source_tree.selection():
            self.remove_button.config(state='normal')
            self.rename_button.config(state='normal')
        else:
            self.remove_button.config(state='disabled')
            self.rename_button.config(state='disabled')
            
    # ========== ACTIONS ==========
    
    def add_sources(self):
        """Add audio source files."""
        files = filedialog.askopenfilenames(
            title="Select Audio Files",
            filetypes=[
                ("Audio Files", "*.wav *.mp3 *.flac *.ogg *.m4a"),
                ("WAV files", "*.wav"),
                ("MP3 files", "*.mp3"),
                ("FLAC files", "*.flac"),
                ("All files", "*.*")
            ]
        )

        if not files:
            return

        # Show loading dialog if multiple files
        loading_dialog = None
        if len(files) > 1:
            loading_dialog = error_dialogs.LoadingDialog(
                self.window if hasattr(self, 'window') and self.window else None,
                title="Loading Audio Files",
                message=f"Loading {len(files)} audio files..."
            )

        try:
            for i, filepath in enumerate(files):
                if loading_dialog:
                    loading_dialog.update_message(
                        f"Loading file {i+1} of {len(files)}...\n{Path(filepath).name}"
                    )
                self.add_source(filepath)
        finally:
            if loading_dialog:
                loading_dialog.close()

    def _on_files_dropped(self, filepaths):
        """
        Handle files dropped via drag-and-drop.

        Args:
            filepaths: List of file paths that were dropped
        """
        logger.info(f"Files dropped: {len(filepaths)} files")

        # Show loading dialog if multiple files
        loading_dialog = None
        if len(filepaths) > 1:
            loading_dialog = error_dialogs.LoadingDialog(
                self.window if hasattr(self, 'window') and self.window else None,
                title="Loading Dropped Files",
                message=f"Loading {len(filepaths)} audio files..."
            )

        try:
            for i, filepath in enumerate(filepaths):
                if loading_dialog:
                    loading_dialog.update_message(
                        f"Loading file {i+1} of {len(filepaths)}...\n{Path(filepath).name}"
                    )
                self.add_source(filepath)
        finally:
            if loading_dialog:
                loading_dialog.close()

    def add_source(self, filepath):
        """Add a single audio source."""
        path = Path(filepath)
        default_name = path.stem
        
        # Ask user for custom name with dialog positioned above this panel
        name = self._show_add_source_dialog(default_name)
        
        # Use default if user cancels or enters empty string
        if not name or not name.strip():
            name = default_name
        
        # Create impressive default positioning - arrange sources in a circle
        num_sources = len(self.sources)
        angle = (num_sources * 60) % 360  # 60 degrees apart, wrapping around
        radius = 3.0  # 3 meters from center
        
        # Calculate position on circle for immediate spatial impression
        import math
        x = radius * math.cos(math.radians(angle))
        z = radius * math.sin(math.radians(angle))
        y = 0.0  # Start at listener height
        
        # Add slight elevation variation for more interest
        if num_sources % 3 == 1:
            y = 1.0  # Some sources slightly above
        elif num_sources % 3 == 2:
            y = -0.5  # Some slightly below
            
        default_position = (round(x, 1), round(y, 1), round(z, 1))
        
        # Use THE method - add_audio_to_project does EVERYTHING
        source_id = self.audio_bridge.add_audio_to_project(
            filepath=filepath,
            position=default_position,
            name=name
        )
        
        if not source_id:
            logger.error(f"Failed to load {filepath}")
            return
        
        # Don't manually add to tree - let SOURCE_ADDED event handler do it
        # This prevents duplicates since audio_bridge.add_audio_to_project emits SOURCE_ADDED

        # Update duration in main window
        if hasattr(self.audio_bridge, 'main_window') and self.audio_bridge.main_window:
            self.audio_bridge.main_window.update_duration()

        # Trigger tree refresh to show the new source
        self.smart_refresh_tree()

        # Invalidate cache
        state_manager.invalidate_cache({f"{self.component_name}_source_list"})
        
        self._log_update(f"Added source: {name}")
        self._update_button_states()
        
    def remove_source(self):
        """Remove selected source."""
        selection = self.source_tree.selection()
        if selection:
            tree_id = selection[0]
            
            # Find the source
            source_to_remove = None
            for source in self.sources:
                if source['tree_id'] == tree_id:
                    source_to_remove = source
                    break
            
            if source_to_remove:
                source_id = source_to_remove['id']
                
                # Remove from audio bridge
                self.audio_bridge.remove_source(source_id)
                
                # Remove from spatial view
                if self.spatial_view:
                    self.spatial_view.remove_source(source_id)

                # Clean up mute/solo states
                self.muted_sources.discard(source_id)
                self.soloed_sources.discard(source_id)

                # Clear selected source if it was the removed one
                if self.selected_source_id == source_id:
                    self.selected_source_id = None

                # Emit state change
                self.emit_state_change(StateEvent.SOURCE_REMOVED, source_id)

                # Refresh tree to reflect removal (avoids stale tree_id issues)
                self.smart_refresh_tree()

                # Invalidate cache
                state_manager.invalidate_cache({f"{self.component_name}_source_list"})
            
            self._log_update(f"Removed source: {source_id}")
            self._update_button_states()
            
    def clear_sources(self):
        """Clear all sources."""
        # Emit events for each source removal
        for source in self.sources:
            self.emit_state_change(StateEvent.SOURCE_REMOVED, source['id'])
            
        self.source_tree.delete(*self.source_tree.get_children())
        self.sources.clear()
        
        # Clear mute/solo states
        self.muted_sources.clear()
        self.soloed_sources.clear()
        self.selected_source_id = None
        
        # Invalidate cache
        state_manager.invalidate_cache({f"{self.component_name}_source_list"})
        
        self._log_update("Cleared all sources")
        self._update_button_states()
        
    def update_source_position(self, source_id, position):
        """Update source position from spatial view."""
        # This now happens through state events
        self.emit_state_change(StateEvent.SOURCE_POSITION_CHANGED, source_id, {
            'position': position
        })
        
    def on_select(self, event):
        """Handle source selection."""
        selection = self.source_tree.selection()
        if selection:
            self.remove_button.config(state='normal')
            
            # Find selected source and emit selection event
            tree_id = selection[0]
            for source in self.sources:
                if source['tree_id'] == tree_id:
                    # Store the selected source ID
                    self.selected_source_id = source['id']
                    
                    # Enable rename button for selected source
                    self.rename_button.config(state='normal')
                    
                    # Notify spatial view through state
                    self.emit_state_change(StateEvent.SELECTION_CHANGED, source['id'], {
                        'component': 'audio_sources_panel',
                        'source_id': source['id']
                    })
                    
                    # Direct notification for immediate response
                    if self.spatial_view:
                        self.spatial_view.select_source(source['id'])
                    break
        else:
            # No selection - disable buttons and clear selected source
            self.selected_source_id = None
            self.remove_button.config(state='disabled')
            self.rename_button.config(state='disabled')
            # solo_button and mute_button moved to Source Control Panel
            
    # Mute/Solo functionality moved to Source Control Panel
        
    # Mute/Solo action methods moved to Source Control Panel
        
    def rename_selected_source(self):
        """Rename the currently selected source."""
        if not self.selected_source_id:
            return
            
        # Find the selected source
        source_to_rename = None
        for source in self.sources:
            if source['id'] == self.selected_source_id:
                source_to_rename = source
                break
                
        if not source_to_rename:
            return
            
        # Get new name from user with dialog positioned above this panel
        current_name = source_to_rename['name']
        new_name = self._show_rename_dialog(current_name)
        
        # Check if user cancelled or entered empty/whitespace name
        if not new_name or not new_name.strip():
            return
            
        new_name = new_name.strip()
        
        # Check if name actually changed
        if new_name == current_name:
            return
            
        # Check for duplicate names
        for source in self.sources:
            if source['id'] != self.selected_source_id and source['name'] == new_name:
                self._show_warning_dialog(
                    "Duplicate Name",
                    f"A source named '{new_name}' already exists. Please choose a different name."
                )
                return
                
        # Update the source name
        old_name = source_to_rename['name']
        source_to_rename['name'] = new_name
        
        # Update the tree display
        if 'tree_id' in source_to_rename and self.source_tree.exists(source_to_rename['tree_id']):
            self.source_tree.item(source_to_rename['tree_id'], text=new_name)
            
        # Update the audio bridge if it tracks names
        if hasattr(self.audio_bridge, 'rename_source'):
            self.audio_bridge.rename_source(self.selected_source_id, new_name)
            
        # Update spatial view if available
        if self.spatial_view and hasattr(self.spatial_view, 'rename_source'):
            self.spatial_view.rename_source(self.selected_source_id, new_name)
            
        # Emit state change event
        self.emit_state_change(StateEvent.SOURCE_RENAMED, self.selected_source_id, {
            'old_name': old_name,
            'new_name': new_name
        })
        
        # Invalidate cache
        state_manager.invalidate_cache({f"{self.component_name}_source_list"})
        
        logger.info(f"Renamed source '{old_name}' to '{new_name}'")
        
    def on_double_click(self, event):
        """Handle double-click on source tree for quick rename."""
        # Get the item that was double-clicked
        item = self.source_tree.identify('item', event.x, event.y)
        if item:
            # Select the item first
            self.source_tree.selection_set(item)
            self.source_tree.focus(item)
            
            # Trigger selection event to update self.selected_source_id
            self.on_select(None)
            
            # Then rename
            self.rename_selected_source()
        
    # Helper methods for mute/solo moved to Source Control Panel
            
    def get_source_mute_state(self, source_id):
        """Get the mute state of a specific source."""
        return source_id in self.muted_sources
        
    def get_source_solo_state(self, source_id):
        """Get the solo state of a specific source."""
        return source_id in self.soloed_sources
        
    def get_all_muted_sources(self):
        """Get set of all muted source IDs."""
        return self.muted_sources.copy()
        
    def get_all_soloed_sources(self):
        """Get set of all soloed source IDs."""
        return self.soloed_sources.copy()

    def format_duration(self, seconds):
        """Format duration as M:SS."""
        minutes = int(seconds // 60)
        secs = int(seconds % 60)
        return f"{minutes}:{secs:02d}"
        
    # ========== UTILITY METHODS ==========
    
    def manual_refresh_sources(self):
        """Manually refresh the source tree - useful for UI consistency and bug prevention."""
        # Save current selection if any
        current_selection = None
        selected_items = self.source_tree.selection()
        if selected_items:
            # Get the source ID from the selected item
            current_selection = selected_items[0]
            
        # Call the existing smart refresh
        self.smart_refresh_tree()
        
        # Try to restore selection if it still exists
        if current_selection and self.source_tree.exists(current_selection):
            self.source_tree.selection_set(current_selection)
            self.source_tree.see(current_selection)
            
        logger.debug(f"Manually refreshed audio sources panel - {len(self.sources)} sources")
    
    def _log_state_event(self, event_type, source_id=None):
        """Log state event reception."""
        msg = f"📡 {event_type}"
        if source_id:
            msg += f" ({source_id})"
        logger.debug(msg)
        
    def _show_rename_dialog(self, current_name):
        """Show rename dialog positioned above the audio sources panel."""
        # Create a custom dialog positioned relative to this panel
        dialog = tk.Toplevel(self.window)
        dialog.title("Rename Source")
        dialog.geometry("400x120")
        dialog.resizable(False, False)
        dialog.transient(self.window)
        
        # Position dialog above this panel
        if self.window:
            # Get panel position
            panel_x = self.window.winfo_x()
            panel_y = self.window.winfo_y()
            panel_width = self.window.winfo_width()
            
            # Center dialog horizontally above panel, offset vertically up
            dialog_x = panel_x + (panel_width - 400) // 2
            dialog_y = max(50, panel_y - 150)  # 150px above panel, min 50px from top
            
            dialog.geometry(f"400x120+{dialog_x}+{dialog_y}")
        
        # Wait for dialog to be properly positioned and visible before grab
        dialog.update_idletasks()
        dialog.grab_set()
        
        # Dialog content
        main_frame = ttk.Frame(dialog, padding="20")
        main_frame.pack(fill='both', expand=True)
        
        # Label
        ttk.Label(main_frame, text="Enter new name for source:").pack(anchor='w', pady=(0, 10))
        
        # Entry field
        entry_var = tk.StringVar(value=current_name)
        entry = ttk.Entry(main_frame, textvariable=entry_var, width=40)
        entry.pack(fill='x', pady=(0, 15))
        entry.select_range(0, 'end')  # Select all text
        entry.focus_set()  # Focus on entry
        
        # Button frame
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill='x')
        
        result = {'value': None}
        
        def on_ok():
            result['value'] = entry_var.get()
            dialog.destroy()
            
        def on_cancel():
            result['value'] = None
            dialog.destroy()
            
        # Buttons
        ttk.Button(button_frame, text="Cancel", command=on_cancel).pack(side='right', padx=(5, 0))
        ttk.Button(button_frame, text="OK", command=on_ok).pack(side='right')
        
        # Bind Enter and Escape keys
        dialog.bind('<Return>', lambda e: on_ok())
        dialog.bind('<Escape>', lambda e: on_cancel())
        
        # Wait for dialog to close
        dialog.wait_window()
        
        return result['value']
        
    def _show_add_source_dialog(self, default_name):
        """Show add source name dialog positioned above the audio sources panel."""
        # Create a custom dialog positioned relative to this panel
        dialog = tk.Toplevel(self.window)
        dialog.title("Source Name")
        dialog.geometry("400x140")
        dialog.resizable(False, False)
        dialog.transient(self.window)
        
        # Position dialog above this panel
        if self.window:
            # Get panel position
            panel_x = self.window.winfo_x()
            panel_y = self.window.winfo_y()
            panel_width = self.window.winfo_width()
            
            # Center dialog horizontally above panel, offset vertically up
            dialog_x = panel_x + (panel_width - 400) // 2
            dialog_y = max(50, panel_y - 170)  # 170px above panel, min 50px from top
            
            dialog.geometry(f"400x140+{dialog_x}+{dialog_y}")
        
        # Wait for dialog to be properly positioned and visible before grab
        dialog.update_idletasks()
        dialog.grab_set()
        
        # Dialog content
        main_frame = ttk.Frame(dialog, padding="20")
        main_frame.pack(fill='both', expand=True)
        
        # Label
        ttk.Label(main_frame, text="Name for this audio source:").pack(anchor='w', pady=(0, 10))
        
        # Entry field
        entry_var = tk.StringVar(value=default_name)
        entry = ttk.Entry(main_frame, textvariable=entry_var, width=40)
        entry.pack(fill='x', pady=(0, 15))
        entry.select_range(0, 'end')  # Select all text
        entry.focus_set()  # Focus on entry
        
        # Button frame
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill='x')
        
        result = {'value': None}
        
        def on_ok():
            result['value'] = entry_var.get()
            dialog.destroy()
            
        def on_cancel():
            result['value'] = None
            dialog.destroy()
            
        # Buttons
        ttk.Button(button_frame, text="Cancel", command=on_cancel).pack(side='right', padx=(5, 0))
        ttk.Button(button_frame, text="OK", command=on_ok).pack(side='right')
        
        # Bind Enter and Escape keys
        dialog.bind('<Return>', lambda e: on_ok())
        dialog.bind('<Escape>', lambda e: on_cancel())
        
        # Wait for dialog to close
        dialog.wait_window()
        
        return result['value']
        
    def _show_warning_dialog(self, title, message):
        """Show warning dialog positioned above the audio sources panel."""
        # Create a custom warning dialog positioned relative to this panel
        dialog = tk.Toplevel(self.window)
        dialog.title(title)
        dialog.geometry("400x150")
        dialog.resizable(False, False)
        dialog.transient(self.window)
        
        # Position dialog above this panel
        if self.window:
            # Get panel position
            panel_x = self.window.winfo_x()
            panel_y = self.window.winfo_y()
            panel_width = self.window.winfo_width()
            
            # Center dialog horizontally above panel, offset vertically up
            dialog_x = panel_x + (panel_width - 400) // 2
            dialog_y = max(50, panel_y - 180)  # 180px above panel, min 50px from top
            
            dialog.geometry(f"400x150+{dialog_x}+{dialog_y}")
        
        # Wait for dialog to be properly positioned and visible before grab
        dialog.update_idletasks()
        dialog.grab_set()
        
        # Dialog content
        main_frame = ttk.Frame(dialog, padding="20")
        main_frame.pack(fill='both', expand=True)
        
        # Warning icon and message
        content_frame = ttk.Frame(main_frame)
        content_frame.pack(fill='both', expand=True, pady=(0, 15))
        
        # Warning label (using emoji since tkinter doesn't have built-in warning icon)
        ttk.Label(content_frame, text="⚠️", font=('TkDefaultFont', 16)).pack(side='left', padx=(0, 10))
        ttk.Label(content_frame, text=message, wraplength=320).pack(side='left', fill='both', expand=True)
        
        # Button frame
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill='x')
        
        def on_ok():
            dialog.destroy()
            
        # OK button
        ttk.Button(button_frame, text="OK", command=on_ok).pack(anchor='center')
        
        # Bind Enter and Escape keys
        dialog.bind('<Return>', lambda e: on_ok())
        dialog.bind('<Escape>', lambda e: on_ok())
        
        # Wait for dialog to close
        dialog.wait_window()
        
    def _log_update(self, reason):
        """Log a necessary update."""
        self._update_count += 1
        logger.debug(f"Update: {reason}")
        
    def _log_avoided_update(self, reason):
        """Log an avoided unnecessary update."""
        self._avoided_updates += 1
        logger.debug(f"Avoided: {reason}")
        
    def _update_performance_display(self):
        """Update performance display periodically."""
        if hasattr(self, 'perf_label') and self.is_visible:
            total = self._update_count + self._avoided_updates
            if total > 0:
                efficiency = self._avoided_updates / total
                self.perf_label.config(
                    text=f"Efficiency: {efficiency:.0%} updates avoided"
                )
            
            # Schedule next update
            if self.window:
                self.window.after(5000, self._update_performance_display)