"""
Immersive Spatial Player Panel

A beautiful, full-screen spatial audio experience built in Python.
Matches the elegance of the web player with native performance.
"""

import tkinter as tk
from tkinter import ttk, font
import numpy as np
import math
import threading
import time
from pathlib import Path
import sys
import logging

# Set up logger
logger = logging.getLogger(__name__)

# Gamepad support - deferred loading
PYGAME_AVAILABLE = None
pygame = None

sys.path.append(str(Path(__file__).parent.parent))
from ..dockable_panel import DockablePanel
from .. import theme
from ..icons import get_icon
from ..theme_helpers import create_themed_button

sys.path.append(str(Path(__file__).parent.parent.parent))
from core.native_spatial_player import NativeSpatialAudioEngine
from core.state_manager import StatefulComponent, StateEvent, state_manager

class SpatialPlayer(DockablePanel, StatefulComponent):
    """Revolutionary spatial audio player with immersive 3D visualization."""
    
    def __init__(self, audio_engine, parent_window=None):
        DockablePanel.__init__(self, "spatial_player", "Spatial Audio Player", parent_window)
        StatefulComponent.__init__(self, "SpatialPlayer")
        
        self.audio_engine = audio_engine
        self.tk_parent = parent_window
        
        # Native spatial audio engine
        self.spatial_engine = NativeSpatialAudioEngine()
        
        # Player state
        self.sources = {}
        self.listener_pos = {'x': 0, 'y': 0, 'z': 0}
        self.listener_rot = {'azimuth': 0, 'elevation': 0}
        self.is_playing = False
        self.current_atmosphere = 'explorer'
        
        # Movement
        self.keys_pressed = set()
        self.mouse_dragging = False
        self.last_mouse_x = 0
        self.last_mouse_y = 0
        
        # Visualization - match web player scale
        self.scale = 50  # pixels per meter - bigger, smoother visuals matching web player
        self.time = 0
        self.animation_running = False
        self.needs_redraw = True  # Flag for event-driven rendering
        self.animation_frame_scheduled = False  # Prevent duplicate frame scheduling
        
        # Performance tracking
        self._frames_rendered = 0
        self._frames_avoided = 0
        self._last_render_time = 0
        
        # Gamepad support
        self.gamepad_connected = False
        self.gamepad = None
        self.gamepad_button_states = {}
        self.gamepad_notification_label = None
        self._setup_gamepad()
        
        # Colors matching web player
        self.colors = {
            'bg': '#0a0a0a',
            'primary': '#00d4ff',
            'secondary': '#ff00ff',
            'text': '#ffffff',
            'text_secondary': '#b0b0b0',
            'grid': '#1a1a1a',
            'grid_major': '#2a2a2a'
        }
        
        # Default to fullscreen-ish size
        self.default_width = 1200
        self.default_height = 800
        
        # Atmosphere presets (simplified for clarity)
        self.atmospheres = {
            'natural': {'name': 'Natural', 'speed': 0.15, 'rotation': 1.0},
            'dynamic': {'name': 'Dynamic', 'speed': 0.25, 'rotation': 2.0},
            'immersive': {'name': 'Immersive', 'speed': 0.2, 'rotation': 1.5}
        }
        self.current_atmosphere = 'natural'  # Start with most intuitive
        
        # Subscribe to state events
        self._setup_state_subscriptions()
        
    def _setup_state_subscriptions(self):
        """Set up state event subscriptions for event-driven rendering."""
        # Source events that might change visualization
        self.subscribe_to_state(StateEvent.SOURCE_POSITION_CHANGED, 'on_source_position_changed')
        self.subscribe_to_state(StateEvent.SOURCE_ADDED, 'on_sources_changed')
        self.subscribe_to_state(StateEvent.SOURCE_REMOVED, 'on_sources_changed')
        
        # Playback events
        self.subscribe_to_state(StateEvent.PLAYBACK_STARTED, 'on_playback_state_changed')
        self.subscribe_to_state(StateEvent.PLAYBACK_STOPPED, 'on_playback_state_changed')
        self.subscribe_to_state(StateEvent.PLAYBACK_POSITION_CHANGED, 'on_playback_position_changed')
        
        logger.debug(f"{self.component_name} subscribed to state events for event-driven rendering")
        
    def setup_content(self):
        """Set up the immersive player interface."""
        # Create transport control icons
        self.play_icon = get_icon('play', size=20, color=self.colors['primary'])
        self.pause_icon = get_icon('pause', size=20, color=self.colors['primary'])
        self.stop_icon = get_icon('stop', size=20, color=self.colors['primary'])

        # Create a regular Frame with dark background inside the ttk content_frame
        self.bg_frame = tk.Frame(self.content_frame, bg=self.colors['bg'])
        self.bg_frame.pack(fill='both', expand=True)

        # Main canvas for 3D visualization (fills entire panel)
        self.canvas = tk.Canvas(
            self.bg_frame,
            bg='#1a1a2e',  # Darker blue instead of pure black for debugging
            highlightthickness=0,
            cursor='crosshair',
            takefocus=True
        )
        self.canvas.pack(fill='both', expand=True)
        
        # Create individual control frames directly on the canvas (no covering overlay)
        # We'll place controls directly without a full-covering frame
        
        # Create floating controls
        self.create_title_display()
        self.create_top_controls()
        self.create_atmosphere_selector()
        self.create_transport_controls()
        self.create_position_display()
        
        # Bind events
        self.setup_interactions()
        
        # Request initial render
        self.request_render()
        
    def create_title_display(self):
        """Create centered title display - moved down to not cover coordinate labels."""
        title_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        title_frame.place(relx=0.5, y=60, anchor='n')  # Moved from y=20 to y=60
        
        # Track title
        self.title_label = tk.Label(
            title_frame,
            text="Welcome to Spatial Audio",
            font=('Arial', 28, 'bold'),
            fg=self.colors['primary'],
            bg=self.colors['bg']
        )
        self.title_label.pack()
        
        # Subtitle
        self.subtitle_label = tk.Label(
            title_frame,
            text="Add audio sources to begin your journey through 3D sound",
            font=('Arial', 14),
            fg=self.colors['text'],
            bg=self.colors['bg']
        )
        self.subtitle_label.pack()
        
    def create_top_controls(self):
        """Create top-right control buttons."""
        controls_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        controls_frame.place(relx=1.0, y=20, anchor='ne', x=-20)
        
        # Fullscreen button
        self.fullscreen_btn = self.create_icon_button(
            controls_frame, 
            "⛶",  # Fullscreen icon 
            self.toggle_fullscreen
        )
        self.fullscreen_btn.pack(side='left', padx=5)
        
        # Refresh sources button
        self.refresh_btn = self.create_icon_button(
            controls_frame,
            "↻",  # Refresh icon
            self.refresh_sources
        )
        self.refresh_btn.pack(side='left', padx=5)
        
        # Info button
        self.info_btn = self.create_icon_button(
            controls_frame,
            "ⓘ",  # Info icon
            self.show_info
        )
        self.info_btn.pack(side='left', padx=5)
        
    def create_atmosphere_selector(self):
        """Create atmosphere preset selector - moved away from compass."""
        atmosphere_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        atmosphere_frame.place(x=20, y=200)  # Moved down from y=80 to not overlap compass
        
        # Label
        tk.Label(
            atmosphere_frame,
            text="Atmosphere",
            font=('Arial', 10),
            fg=self.colors['text_secondary'],
            bg=self.colors['bg']
        ).pack()
        
        # Current atmosphere button
        self.atmosphere_btn = tk.Button(
            atmosphere_frame,
            text=self.atmospheres[self.current_atmosphere]['name'],
            font=('Arial', 14, 'bold'),
            fg=self.colors['primary'],
            bg='#1a1a1a',
            bd=0,
            padx=20,
            pady=10,
            cursor='hand2',
            command=self.cycle_atmosphere
        )
        self.atmosphere_btn.pack()
        
    def create_transport_controls(self):
        """Create bottom transport controls."""
        transport_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        transport_frame.place(relx=0.5, rely=1.0, anchor='s', y=-30)

        # Play/Pause button
        self.play_pause_btn = self.create_transport_button(
            transport_frame,
            " Play",
            self.toggle_playback,
            icon=self.play_icon
        )
        self.play_pause_btn.pack(side='left', padx=10)

        # Stop button
        self.stop_btn = self.create_transport_button(
            transport_frame,
            " Stop",
            self.stop_playback,
            icon=self.stop_icon
        )
        self.stop_btn.pack(side='left', padx=10)

        # Reset position button
        self.reset_btn = self.create_transport_button(
            transport_frame,
            " Reset",
            self.reset_position
        )
        self.reset_btn.pack(side='left', padx=10)
        
    def create_position_display(self):
        """Create listener position display."""
        pos_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        pos_frame.place(x=20, rely=1.0, anchor='sw', y=-30)
        
        self.position_label = tk.Label(
            pos_frame,
            text="X: 0.0m | Y: +0.0m (height) | Z: 0.0m",
            font=('Arial', 10),
            fg=self.colors['text_secondary'],
            bg=self.colors['bg']
        )
        self.position_label.pack()
        
        # Add controls help in bottom center - dynamic based on gamepad connection
        controls_frame = tk.Frame(self.bg_frame, bg=self.colors['bg'])
        controls_frame.place(relx=0.5, rely=1.0, anchor='s', y=-10)
        
        # Create dynamic controls label
        self.controls_label = tk.Label(
            controls_frame,
            text="",  # Will be updated by update_controls_display()
            font=('Arial', 8),
            fg='#606060',
            bg=self.colors['bg']
        )
        self.controls_label.pack()
        
        # Initialize controls display
        self.update_controls_display()
        
        # Create gamepad notification area
        self.gamepad_notification_label = tk.Label(
            self.bg_frame,
            text="",
            font=('Arial', 9),
            fg='#00d4ff',
            bg=self.colors['bg']
        )
        self.gamepad_notification_label.place(relx=1.0, rely=0.0, anchor='ne', x=-10, y=50)
        
    def create_icon_button(self, parent, text, command):
        """Create a styled icon button."""
        btn = tk.Button(
            parent,
            text=text,
            font=('Arial', 16),
            fg=self.colors['text'],
            bg='#1a1a1a',
            bd=0,
            padx=10,
            pady=5,
            cursor='hand2',
            command=command
        )
        
        # Hover effects
        def on_enter(e):
            btn.config(bg='#2a2a2a', fg=self.colors['primary'])
        def on_leave(e):
            btn.config(bg='#1a1a1a', fg=self.colors['text'])
            
        btn.bind('<Enter>', on_enter)
        btn.bind('<Leave>', on_leave)
        
        return btn
        
    def create_transport_button(self, parent, text, command, icon=None):
        """Create a transport control button with optional icon."""
        btn_config = {
            'text': text,
            'font': ('Arial', 12),
            'fg': self.colors['primary'],
            'bg': '#1a1a1a',
            'bd': 0,
            'padx': 15,
            'pady': 10,
            'cursor': 'hand2',
            'command': command
        }

        # Add icon if provided
        if icon:
            btn_config['image'] = icon
            btn_config['compound'] = 'left'

        btn = tk.Button(parent, **btn_config)

        # Keep reference to icon to prevent garbage collection
        if icon:
            btn.image = icon

        # Hover effects
        def on_enter(e):
            btn.config(bg=self.colors['primary'], fg=self.colors['bg'])
        def on_leave(e):
            btn.config(bg='#1a1a1a', fg=self.colors['primary'])

        btn.bind('<Enter>', on_enter)
        btn.bind('<Leave>', on_leave)

        return btn
        
    def setup_interactions(self):
        """Set up mouse and keyboard interactions."""
        # Mouse controls
        self.canvas.bind('<Button-1>', self.on_mouse_down)
        self.canvas.bind('<B1-Motion>', self.on_mouse_drag)
        self.canvas.bind('<ButtonRelease-1>', self.on_mouse_up)
        self.canvas.bind('<MouseWheel>', self.on_mouse_wheel)
        self.canvas.bind('<Button-4>', lambda e: self.on_mouse_wheel(e, delta=120))
        self.canvas.bind('<Button-5>', lambda e: self.on_mouse_wheel(e, delta=-120))
        
        # Keyboard controls - focus on canvas for key events
        self.canvas.focus_set()
        self.canvas.bind('<KeyPress>', self.on_key_press)
        self.canvas.bind('<KeyRelease>', self.on_key_release)
        
        # Canvas resize
        self.canvas.bind('<Configure>', self.on_canvas_resize)
        
    def request_render(self):
        """Request a render frame - avoids duplicate scheduling."""
        if not hasattr(self, 'canvas') or not self.canvas:
            # Canvas not ready yet, skip render
            return
            
        if not self.animation_frame_scheduled:
            self.animation_frame_scheduled = True
            self.canvas.after_idle(self.render_frame)
            
    def render_frame(self):
        """Render a single frame only when needed."""
        self.animation_frame_scheduled = False
        
        # Track performance
        self._frames_rendered += 1
        current_time = time.time()
        
        # Log performance every 30 seconds
        if current_time - self._last_render_time > 30:
            if self._frames_rendered > 0:
                efficiency = (self._frames_avoided / (self._frames_rendered + self._frames_avoided)) * 100
                logger.debug(f"Immersive Player Performance: {self._frames_rendered} frames rendered, "
                      f"{self._frames_avoided} avoided ({efficiency:.1f}% efficiency)")
            self._frames_rendered = 0
            self._frames_avoided = 0
            self._last_render_time = current_time
            
        # Clear canvas
        self.canvas.delete('all')
        
        # Get canvas size
        w = self.canvas.winfo_width()
        h = self.canvas.winfo_height()
        
        # Skip if canvas not ready yet
        if w <= 1 or h <= 1:
            self.canvas.after(50, self.request_render)
            return
            
        cx = w // 2
        cy = h // 2
        
        # Draw coordinate system with proper labels
        self.canvas.create_line(0, cy, w, cy, fill='#ff4444', width=2)  # X axis (horizontal)
        self.canvas.create_line(cx, 0, cx, h, fill='#44ff44', width=2)  # Z axis (vertical, forward=up)
        
        # Coordinate labels for clarity (corrected: Z is forward/back, Y is elevation)
        self.canvas.create_text(w-40, cy-15, text="RIGHT (+X)", fill='#ff4444', font=('Arial', 10, 'bold'))
        self.canvas.create_text(40, cy-15, text="LEFT (-X)", fill='#ff4444', font=('Arial', 10, 'bold'))
        self.canvas.create_text(cx+50, 20, text="FRONT (+Z)", fill='#44ff44', font=('Arial', 10, 'bold'))
        self.canvas.create_text(cx+50, h-20, text="BACK (-Z)", fill='#44ff44', font=('Arial', 10, 'bold'))
        
        # Center crosshairs only (no confusing yellow dot)
        # The actual listener is drawn by draw_listener() method
        
        # Simple elevation legend in corner  
        self.canvas.create_text(w-100, h-60, text="ELEVATION (Y):", fill='white', font=('Arial', 9, 'bold'))
        self.canvas.create_text(w-100, h-45, text="↑ Above (+Y)", fill='#88ff88', font=('Arial', 8))
        self.canvas.create_text(w-100, h-30, text="↓ Below (-Y)", fill='#ff8888', font=('Arial', 8))
        self.canvas.create_text(w-100, h-15, text="Dotted line = height", fill='#cccccc', font=('Arial', 8))
        
        # Draw grid (simplified)
        try:
            self.draw_grid_simple(w, h, cx, cy)
        except Exception as e:
            logger.error(f"Grid drawing error: {e}")
        
        # Draw listener sprite (moving pink circle)
        try:
            self.draw_listener(cx, cy)
        except Exception as e:
            logger.error(f"Listener drawing error: {e}")
            
        # Draw sources (simplified)
        try:
            self.draw_sources_simple(cx, cy)
        except Exception as e:
            logger.error(f"Sources drawing error: {e}")
            
        # Draw compass for listening direction
        try:
            self.draw_compass()
        except Exception as e:
            logger.error(f"Compass drawing error: {e}")
        
        # Update movement
        self.update_movement()
        
        # Update position display with clear labels
        self.position_label.config(
            text=f"X: {self.listener_pos['x']:.1f}m | Y: {self.listener_pos['y']:+.1f}m (elevation) | Z: {self.listener_pos['z']:.1f}m"
        )
        
        # Only schedule continuous updates if actively moving or playing
        if self.keys_pressed or self.mouse_dragging or self.is_playing:
            # Schedule next frame for smooth movement
            self.canvas.after(33, self.request_render)
        else:
            # Track avoided frame
            self._frames_avoided += 1
            
    # State event handlers for event-driven rendering
    def on_source_position_changed(self, event):
        """Handle source position changes."""
        self.request_render()
        
    def on_sources_changed(self, event):
        """Handle source additions/removals."""
        self.refresh_sources()
        self.request_render()
        
    def on_playback_state_changed(self, event):
        """Handle playback start/stop."""
        self.is_playing = (event.event_type == StateEvent.PLAYBACK_STARTED)
        self.request_render()
        
    def on_playback_position_changed(self, event):
        """Handle playback position updates."""
        # Only render if we're actually playing
        if self.is_playing:
            self.request_render()
        
    def draw_background_effects(self, w, h):
        """Draw subtle background effects - minimal to keep focus on audio."""
        # Single subtle pulse at center when no sources loaded
        if not self.sources:
            cx, cy = w // 2, h // 2
            pulse = 0.5 + 0.5 * math.sin(self.time * 2)
            r = 100 * pulse
            
            # Gentle invitation glow
            for j in range(5):
                alpha = 0.03 * (1 - j/5) * pulse
                fill = self.hex_with_alpha(self.colors['primary'], alpha)
                
                self.canvas.create_oval(
                    cx - r*(j+1)/5, cy - r*(j+1)/5,
                    cx + r*(j+1)/5, cy + r*(j+1)/5,
                    fill=fill, outline='', tags='bg'
                )
                
    def draw_grid(self, w, h, cx, cy):
        """Draw spatial grid matching web player's X-Z layout."""
        grid_spacing = self.scale
        
        # Calculate visible range
        x_start = int(-cx / grid_spacing) - 1
        x_end = int((w - cx) / grid_spacing) + 1
        y_start = int(-cy / grid_spacing) - 1
        y_end = int((h - cy) / grid_spacing) + 1
        
        # Draw grid lines
        for i in range(x_start, x_end):
            x = cx + i * grid_spacing
            color = self.colors['grid_major'] if i % 5 == 0 else self.colors['grid']
            width = 2 if i == 0 else 1
            
            self.canvas.create_line(
                x, 0, x, h,
                fill=color, width=width, tags='grid'
            )
            
        for i in range(y_start, y_end):
            y = cy + i * grid_spacing
            color = self.colors['grid_major'] if i % 5 == 0 else self.colors['grid']
            width = 2 if i == 0 else 1
            
            self.canvas.create_line(
                0, y, w, y,
                fill=color, width=width, tags='grid'
            )
            
        # Add axis labels to match web player
        # X axis (left-right)
        self.canvas.create_text(
            w - 30, cy + 20,
            text="East (+X)", fill=self.colors['text_secondary'],
            font=('Arial', 10), tags='grid'
        )
        self.canvas.create_text(
            30, cy + 20,
            text="West (-X)", fill=self.colors['text_secondary'],
            font=('Arial', 10), tags='grid'
        )
        
        # Z axis (forward-back, with forward UP on screen)
        self.canvas.create_text(
            cx + 20, 30,
            text="North (+Z)", fill=self.colors['text_secondary'],
            font=('Arial', 10), tags='grid'
        )
        self.canvas.create_text(
            cx + 20, h - 20,
            text="South (-Z)", fill=self.colors['text_secondary'],
            font=('Arial', 10), tags='grid'
        )
            
    def draw_distance_circles(self, cx, cy):
        """Draw distance reference circles."""
        distances = [2, 5, 10]  # meters
        
        for d in distances:
            r = d * self.scale
            
            # Draw circle
            self.canvas.create_oval(
                cx - r, cy - r, cx + r, cy + r,
                outline='#333333', width=1, dash=(5, 5), tags='distance'
            )
            
            # Label
            self.canvas.create_text(
                cx + r - 10, cy - 10,
                text=f"{d}m", fill=self.colors['text_secondary'],
                font=('Arial', 8), tags='distance'
            )
            
    def draw_sources(self, cx, cy):
        """Draw audio sources with animated effects."""
        for name, source in self.sources.items():
            # Convert to screen coordinates to match web player visualization:
            # Screen X = World X (left/right)
            # Screen Y = -World Z (forward is UP on screen, backward is DOWN)
            # World Y = elevation (shown as visual effects, not position)
            sx = cx + source['position'][0] * self.scale  # X: left(-) to right(+)
            sy = cy - source['position'][2] * self.scale  # Z: forward(+) is UP, back(-) is DOWN
            
            # Source color based on hash
            colors = ['#ff6666', '#66ff66', '#6666ff', '#ffff66', '#ff66ff', '#66ffff']
            color = colors[hash(name) % len(colors)]
            
            # Get Y elevation for visual effects
            elevation = source['position'][1]  # Y: elevation (up/down height)
            
            # Animated pulse effect
            pulse = 1 + 0.1 * math.sin(self.time * 3 + hash(name))
            
            # Size varies with elevation (higher = larger, lower = smaller)
            elevation_scale = 1.0 + (elevation * 0.2)  # +0.2 size per meter elevation
            r = 8 * pulse * max(0.3, elevation_scale)  # Minimum size 30%
            
            # Elevation-based visual effects
            if elevation > 0.1:  # Above listener
                # Brighter/more intense for higher sources
                intensity = min(1.0, 0.5 + elevation * 0.2)
            elif elevation < -0.1:  # Below listener  
                # Dimmer/more transparent for lower sources
                intensity = max(0.3, 1.0 + elevation * 0.2)
            else:
                intensity = 1.0  # At listener level
            
            # Draw elevation shadow/connection if source is elevated
            if abs(elevation) > 0.1:
                # Shadow on ground plane
                shadow_offset = elevation * 5  # 5 pixels per meter
                shadow_alpha = 0.2 * (1.0 - min(abs(elevation) / 10.0, 1.0))
                
                # Connecting line
                self.canvas.create_line(
                    sx, sy + shadow_offset,
                    sx, sy,
                    fill=self.hex_with_alpha('#666666', 0.4),
                    width=1, dash=(3, 2), tags='source'
                )
                
                # Ground shadow
                self.canvas.create_oval(
                    sx - r * 0.7, sy + shadow_offset - r * 0.3,
                    sx + r * 0.7, sy + shadow_offset + r * 0.3,
                    fill=self.hex_with_alpha('#000000', shadow_alpha),
                    outline='', tags='source'
                )
            
            # Glow effect with elevation intensity
            for i in range(3):
                glow_r = r + i * 5
                alpha = (0.3 - i * 0.1) * intensity
                self.canvas.create_oval(
                    sx - glow_r, sy - glow_r,
                    sx + glow_r, sy + glow_r,
                    fill='', outline=self.hex_with_alpha(color, alpha),
                    width=2, tags='source'
                )
            
            # Adjust color intensity based on elevation
            if elevation > 0.1:
                outline_color = '#ffffff'  # White outline for above
                outline_width = 3
            elif elevation < -0.1:
                outline_color = '#888888'  # Gray outline for below
                outline_width = 1
            else:
                outline_color = '#ffffff'  # White for at level
                outline_width = 2
            
            # Main source circle
            self.canvas.create_oval(
                sx - r, sy - r, sx + r, sy + r,
                fill=color, outline=outline_color, width=outline_width, tags='source'
            )
            
            # Source name
            self.canvas.create_text(
                sx, sy - r - 8,
                text=name, fill='white',
                font=('Arial', 10, 'bold'), tags='source'
            )
            
            # Elevation indicator with clear labeling
            if abs(elevation) > 0.1:
                if elevation > 0:
                    elev_text = f"↑{elevation:.1f}m"  # Above
                    elev_color = '#88ff88'
                else:
                    elev_text = f"↓{abs(elevation):.1f}m"  # Below
                    elev_color = '#ff8888'
                
                self.canvas.create_text(
                    sx, sy + r + 15,
                    text=elev_text,
                    fill=elev_color,
                    font=('Arial', 9, 'bold'), tags='source'
                )
                
                
    def on_mouse_down(self, event):
        """Handle mouse down."""
        # Set focus so keyboard controls (WASD) work after clicking
        self.canvas.focus_set()
        self.mouse_dragging = True
        self.last_mouse_x = event.x
        self.last_mouse_y = event.y
        # Start rendering for drag
        self.request_render()
        
    def on_mouse_drag(self, event):
        """Handle mouse drag for rotation."""
        if self.mouse_dragging:
            dx = event.x - self.last_mouse_x
            dy = event.y - self.last_mouse_y
            
            # Update rotation
            rot_speed = self.atmospheres[self.current_atmosphere]['rotation']
            self.listener_rot['azimuth'] -= dx * rot_speed * 0.5
            self.listener_rot['azimuth'] %= 360
            
            # Update elevation (clamped)
            self.listener_rot['elevation'] = max(-90, min(90, 
                self.listener_rot['elevation'] - dy * rot_speed * 0.3))
            
            # Update spatial engine
            self.spatial_engine.update_listener_rotation(
                self.listener_rot['azimuth'],
                self.listener_rot['elevation']
            )
            
            self.last_mouse_x = event.x
            self.last_mouse_y = event.y
            
            # Request render for rotation changes
            self.request_render()
            
    def on_mouse_up(self, event):
        """Handle mouse up."""
        self.mouse_dragging = False
        # Final render after drag
        self.request_render()
        
    def on_mouse_wheel(self, event, delta=None):
        """Handle mouse wheel - zoom only with Ctrl held."""
        # Only zoom if Ctrl is held - keeps default experience focused on spatial audio
        if event.state & 0x4:  # Ctrl key held
            if delta is None:
                delta = event.delta
                
            # Zoom in/out
            if delta > 0:
                self.scale = min(100, self.scale * 1.1)
            else:
                self.scale = max(10, self.scale / 1.1)
                
            # Request render for zoom changes
            self.request_render()
            
    def on_key_press(self, event):
        """Handle key press."""
        key = event.keysym.lower()
        self.keys_pressed.add(key)
        
        # Handle special keys
        if key == 'r':
            self.reset_position()
        elif key == 'space':
            self.toggle_playback()
        # Arrow keys for listening direction (matches web player)
        elif key == 'left':
            self.listener_rot['azimuth'] -= 5  # Turn left
            self.spatial_engine.update_listener_rotation(
                self.listener_rot['azimuth'],
                self.listener_rot['elevation']
            )
        elif key == 'right':
            self.listener_rot['azimuth'] += 5  # Turn right
            self.spatial_engine.update_listener_rotation(
                self.listener_rot['azimuth'],
                self.listener_rot['elevation']
            )
        elif key == 'up':
            self.listener_rot['elevation'] = min(90, self.listener_rot['elevation'] + 5)  # Look up
            self.spatial_engine.update_listener_rotation(
                self.listener_rot['azimuth'],
                self.listener_rot['elevation']
            )
        elif key == 'down':
            self.listener_rot['elevation'] = max(-90, self.listener_rot['elevation'] - 5)  # Look down
            self.spatial_engine.update_listener_rotation(
                self.listener_rot['azimuth'],
                self.listener_rot['elevation']
            )
            
        # Request render when movement starts
        self.request_render()
        
    def on_key_release(self, event):
        """Handle key release."""
        self.keys_pressed.discard(event.keysym.lower())
        # Request final render when movement stops
        if not self.keys_pressed:
            self.request_render()
        
    def update_movement(self):
        """Update listener position based on keyboard input."""
        if not self.keys_pressed:
            return
            
        # Consistent movement speed - no acceleration or speed buildup
        speed = 0.15  # Fixed speed regardless of atmosphere or key press duration
        
        # Calculate movement using CORRECT coordinate mapping:
        # X = left/right
        # Y = up/down (elevation)
        # Z = forward/back
        dx = dy = dz = 0
        
        # WASD for horizontal movement
        if 'w' in self.keys_pressed:
            dz += speed  # W = forward (positive Z)
        if 's' in self.keys_pressed:
            dz -= speed  # S = backward (negative Z)
        if 'a' in self.keys_pressed:
            dx -= speed  # A = left (negative X)
        if 'd' in self.keys_pressed:
            dx += speed  # D = right (positive X)
            
        # Q/E for vertical movement (elevation)
        if 'q' in self.keys_pressed:
            dy += speed  # Q = up (positive Y)
        if 'e' in self.keys_pressed:
            dy -= speed  # E = down (negative Y)
            
        # Movement is ABSOLUTE to world coordinates - NOT rotated by listening direction
        # W/S = North/South, A/D = West/East regardless of where you're looking
        # Update position directly without rotation
        self.listener_pos['x'] += dx  # A/D movement - always east/west
        self.listener_pos['y'] += dy  # Q/E movement - always up/down  
        self.listener_pos['z'] += dz  # W/S movement - always north/south
        
        # Update spatial engine with correct coordinate order
        self.spatial_engine.update_listener_position(
            self.listener_pos['x'],
            self.listener_pos['y'],  # Y is elevation
            self.listener_pos['z']   # Z is forward/back
        )
        
    def on_canvas_resize(self, event):
        """Handle canvas resize."""
        # Request render for new canvas size
        self.request_render()
        
    def toggle_fullscreen(self):
        """Toggle fullscreen mode."""
        # Get the panel window
        window = self.winfo_toplevel()
        current_state = window.attributes('-fullscreen')
        window.attributes('-fullscreen', not current_state)
        
    def refresh_sources(self):
        """Refresh sources from audio sources panel."""
        if not hasattr(self, 'title_label'):
            # UI not ready, do headless refresh
            return self._refresh_sources_headless()
            
        try:
            # Check parent window
            if not hasattr(self.parent_window, 'audio_sources_panel'):
                self.subtitle_label.config(text="Use View menu → Audio Sources to add your music")
                return
                
            sources_metadata = self.parent_window.audio_sources_panel.sources
            if not sources_metadata:
                self.subtitle_label.config(text="Add audio files to create your 3D soundscape")
                return
                
            # Prepare sources for spatial engine
            sources_data = []
            self.sources.clear()
            
            for source in sources_metadata:
                source_id = source.get('id')
                source_name = source.get('name', 'Unknown')
                position = source.get('position', (0, 0, 0))
                
                if source_id in self.parent_window.audio_bridge.loaded_samples:
                    sample_info = self.parent_window.audio_bridge.loaded_samples[source_id]
                    audio_data = sample_info['audio_data']
                    
                    sources_data.append({
                        'name': source_name,
                        'audio_data': audio_data,
                        'position': list(position),
                        'id': source_id
                    })
                    
                    # Store for visualization
                    self.sources[source_name] = {
                        'position': position,
                        'id': source_id
                    }
                    
            if sources_data:
                # Load into spatial engine
                num_loaded = self.spatial_engine.load_sources(sources_data)
                
                # Update UI
                if num_loaded == 1:
                    self.title_label.config(text=sources_data[0]['name'])
                else:
                    self.title_label.config(text=f"{num_loaded} Sources Loaded")
                    
                self.subtitle_label.config(text="Press play and use WASD to walk through your music")
                
                logging.info(f"Loaded {num_loaded} sources into immersive player")
                
        except Exception as e:
            logging.error(f"Failed to refresh sources: {e}")
            self.subtitle_label.config(text="Error loading sources")
            
    def _refresh_sources_headless(self):
        """Refresh sources without UI."""
        try:
            if not hasattr(self.parent_window, 'audio_sources_panel'):
                return 0
                
            sources_metadata = self.parent_window.audio_sources_panel.sources
            if not sources_metadata:
                return 0
                
            sources_data = []
            self.sources.clear()
            
            for source in sources_metadata:
                source_id = source.get('id')
                source_name = source.get('name', 'Unknown')
                position = source.get('position', (0, 0, 0))
                
                if source_id in self.parent_window.audio_bridge.loaded_samples:
                    sample_info = self.parent_window.audio_bridge.loaded_samples[source_id]
                    audio_data = sample_info['audio_data']
                    
                    sources_data.append({
                        'name': source_name,
                        'audio_data': audio_data,
                        'position': list(position),
                        'id': source_id
                    })
                    
                    self.sources[source_name] = {
                        'position': position,
                        'id': source_id
                    }
                    
            if sources_data:
                return self.spatial_engine.load_sources(sources_data)
            return 0
            
        except Exception as e:
            logging.error(f"Headless refresh failed: {e}")
            return 0
            
    def cycle_atmosphere(self):
        """Cycle through atmosphere presets."""
        presets = list(self.atmospheres.keys())
        current_idx = presets.index(self.current_atmosphere)
        self.current_atmosphere = presets[(current_idx + 1) % len(presets)]
        
        # Update button
        self.atmosphere_btn.config(text=self.atmospheres[self.current_atmosphere]['name'])
        
        # Request render for any visual changes related to atmosphere
        self.request_render()
        
    def toggle_playback(self):
        """Toggle play/pause."""
        if self.is_playing:
            self.spatial_engine.pause()
            self.play_pause_btn.config(text=" Play", image=self.play_icon)
            self.is_playing = False
        else:
            self.spatial_engine.play()
            self.play_pause_btn.config(text=" Pause", image=self.pause_icon)
            self.is_playing = True

    def stop_playback(self):
        """Stop playback."""
        self.spatial_engine.stop()
        self.play_pause_btn.config(text=" Play", image=self.play_icon)
        self.is_playing = False
        
    def reset_position(self):
        """Reset listener to origin."""
        self.listener_pos = {'x': 0, 'y': 0, 'z': 0}
        self.listener_rot = {'azimuth': 0, 'elevation': 0}
        
        self.spatial_engine.update_listener_position(0, 0, 0)
        self.spatial_engine.update_listener_rotation(0, 0)
        
        # Request render for position reset
        self.request_render()
        
    def show_info(self):
        """Show info/help."""
        info_text = """IMMERSIVE SPATIAL PLAYER

Movement:
• WASD - Move horizontally
• Q/E - Move up/down
• Mouse drag - Look around
• Scroll - Zoom in/out

Atmosphere Presets:
• Explorer - Fast movement
• Venue - Concert hall feel
• Stadium - Large space
• Asymmetric - Experimental
• Elastic - Smooth movement

Press ESC for fullscreen"""
        
        import tkinter.messagebox as messagebox
        messagebox.showinfo("Spatial Player Help", info_text)
        
    def hex_with_alpha(self, hex_color, alpha):
        """Convert hex color to rgba-like for canvas (returns darker color for alpha effect)."""
        # Simple alpha simulation by darkening the color
        r = int(hex_color[1:3], 16)
        g = int(hex_color[3:5], 16)
        b = int(hex_color[5:7], 16)
        
        r = int(r * alpha)
        g = int(g * alpha)
        b = int(b * alpha)
        
        return f"#{r:02x}{g:02x}{b:02x}"
        
    def draw_grid_simple(self, w, h, cx, cy):
        """Draw simplified grid."""
        spacing = self.scale  # Use scale for grid spacing
        
        # Only draw visible grid lines
        start_x = (cx % spacing) - spacing
        start_y = (cy % spacing) - spacing
        
        # Vertical lines
        x = start_x
        while x < w:
            if x >= 0:
                color = '#444444' if int(x - cx) % (spacing * 5) != 0 else '#666666'
                self.canvas.create_line(x, 0, x, h, fill=color, width=1)
            x += spacing
            
        # Horizontal lines  
        y = start_y
        while y < h:
            if y >= 0:
                color = '#444444' if int(y - cy) % (spacing * 5) != 0 else '#666666'
                self.canvas.create_line(0, y, w, y, fill=color, width=1)
            y += spacing
            
    def draw_listener(self, cx, cy):
        """Draw listener matching web player style - muted pink with sound waves."""
        # HUMAN NAVIGATION VIEW: Forward/back movement shows as UP/DOWN on screen
        x = cx + self.listener_pos['x'] * self.scale  # X: left/right  
        y = cy - self.listener_pos['z'] * self.scale  # Z: forward(+) UP, back(-) DOWN
        elevation = self.listener_pos['y']  # Y: elevation (visual effects)
        
        # Web player style - consistent 12px radius
        radius = 12
        
        # Outer border like web player
        self.canvas.create_oval(
            x - radius - 1, y - radius - 1, 
            x + radius + 1, y + radius + 1,
            fill='', outline='#666666', width=1, tags='listener'
        )
        
        # Main body - web player muted pink
        self.canvas.create_oval(
            x - radius, y - radius, x + radius, y + radius,
            fill='#c85a8e', outline='', tags='listener'
        )
        
        # Sound wave direction indicator (web player style)
        # Azimuth 0 = North (up on screen), 90 = East (right), 180 = South (down), 270 = West (left)
        # Convert to screen coordinates where 0 radians = right
        dir_angle_rad = math.radians(self.listener_rot['azimuth'] - 90)
        
        # Draw 3 concentric sound wave arcs showing listening direction
        for i in range(3):
            wave_radius = 18 + (i * 6)
            arc_length = math.pi / 3  # 60 degree arc
            start_angle = dir_angle_rad - arc_length / 2
            end_angle = dir_angle_rad + arc_length / 2
            
            # Calculate arc points (tkinter doesn't have native arc stroke)
            arc_points = []
            for angle in [start_angle + (end_angle - start_angle) * t / 20 for t in range(21)]:
                arc_x = x + math.cos(angle) * wave_radius
                arc_y = y + math.sin(angle) * wave_radius
                arc_points.extend([arc_x, arc_y])
            
            if len(arc_points) >= 4:
                alpha = 0.6 - i * 0.15
                color = self.hex_with_alpha('#c85a8e', alpha)
                self.canvas.create_line(
                    arc_points, fill=color, width=2, 
                    smooth=True, tags='listener'
                )
        
        # Small directional indicator line for clarity
        dir_length = 20
        dir_x = x + math.cos(dir_angle_rad) * dir_length
        dir_y = y + math.sin(dir_angle_rad) * dir_length
        
        self.canvas.create_line(
            x, y, dir_x, dir_y,
            fill=self.hex_with_alpha('#c85a8e', 0.8), 
            width=2, capstyle='round', tags='listener'
        )
        
        # Show elevation if significant (dotted line like sources)
        if abs(elevation) > 0.1:
            elev_color = '#88ff88' if elevation > 0 else '#ff8888'
            line_length = abs(elevation) * self.scale * 0.5
            line_end_y = y + (line_length if elevation < 0 else -line_length)
            
            # Dotted line showing elevation
            num_dashes = max(3, int(line_length / 5))
            for i in range(num_dashes):
                dash_start_y = y + (line_end_y - y) * i / num_dashes
                dash_end_y = y + (line_end_y - y) * (i + 0.6) / num_dashes
                self.canvas.create_line(
                    x, dash_start_y, x, dash_end_y,
                    fill=elev_color, width=2, tags='listener'
                )
            
    def draw_compass(self):
        """Draw compass for listening direction (matches web player)."""
        # Get canvas dimensions for listening beam
        w = self.canvas.winfo_width()
        h = self.canvas.winfo_height()
        
        # Compass position (top-left like web player)
        cx, cy = 80, 80
        radius = 35
        
        # Convert azimuth to compass direction
        azimuth = self.listener_rot['azimuth']
        while azimuth < 0:
            azimuth += 360
        while azimuth >= 360:
            azimuth -= 360
            
        # Get compass direction
        if azimuth >= 337.5 or azimuth < 22.5:
            direction = 'N'
        elif azimuth < 67.5:
            direction = 'NE'
        elif azimuth < 112.5:
            direction = 'E'
        elif azimuth < 157.5:
            direction = 'SE'
        elif azimuth < 202.5:
            direction = 'S'
        elif azimuth < 247.5:
            direction = 'SW'
        elif azimuth < 292.5:
            direction = 'W'
        else:
            direction = 'NW'
        
        # Background circle with glow
        self.canvas.create_oval(
            cx - radius - 10, cy - radius - 10,
            cx + radius + 10, cy + radius + 10,
            fill=self.hex_with_alpha('#0a0a0a', 0.8),
            outline=self.hex_with_alpha('#00d4ff', 0.6),
            width=3, tags='compass'
        )
        
        # Inner compass circle
        self.canvas.create_oval(
            cx - radius, cy - radius,
            cx + radius, cy + radius,
            fill='',
            outline=self.hex_with_alpha('#00d4ff', 0.8),
            width=2, tags='compass'
        )
        
        # Cardinal direction lines
        for i in range(4):
            angle_rad = (i * math.pi) / 2
            inner_x = cx + math.sin(angle_rad) * (radius - 15)
            inner_y = cy - math.cos(angle_rad) * (radius - 15)
            outer_x = cx + math.sin(angle_rad) * (radius + 5)
            outer_y = cy - math.cos(angle_rad) * (radius + 5)
            
            self.canvas.create_line(
                inner_x, inner_y, outer_x, outer_y,
                fill=self.hex_with_alpha('#ffffff', 0.3),
                width=1, tags='compass'
            )
            
        # Direction indicator (arrow pointing in listening direction)
        angle_rad = -math.radians(self.listener_rot['azimuth']) - math.pi / 2
        tip_x = cx + math.cos(angle_rad) * radius
        tip_y = cy + math.sin(angle_rad) * radius
        
        # Arrow with enhanced visibility
        arrow_size = radius - 20
        left_x = cx + math.cos(angle_rad + 2.5) * arrow_size
        left_y = cy + math.sin(angle_rad + 2.5) * arrow_size
        right_x = cx + math.cos(angle_rad - 2.5) * arrow_size
        right_y = cy + math.sin(angle_rad - 2.5) * arrow_size
        
        # Arrow shadow
        self.canvas.create_polygon(
            [tip_x + 2, tip_y + 2, left_x + 2, left_y + 2, right_x + 2, right_y + 2],
            fill=self.hex_with_alpha('#000000', 0.5),
            outline='', tags='compass'
        )
        
        # Main arrow
        self.canvas.create_polygon(
            [tip_x, tip_y, left_x, left_y, right_x, right_y],
            fill='#00d4ff',
            outline='white',
            width=1, tags='compass'
        )
        
        # Cardinal direction labels
        directions = ['N', 'E', 'S', 'W']
        for i, dir_label in enumerate(directions):
            angle_rad = (i * math.pi) / 2
            label_x = cx + math.sin(angle_rad) * (radius + 20)
            label_y = cy - math.cos(angle_rad) * (radius + 20)
            
            # Highlight current direction
            if dir_label in direction:
                color = '#ffffff'
                font_style = ('Arial', 12, 'bold')
            else:
                color = self.hex_with_alpha('#ffffff', 0.7)
                font_style = ('Arial', 10)
                
            self.canvas.create_text(
                label_x, label_y,
                text=dir_label,
                fill=color,
                font=font_style,
                tags='compass'
            )
            
        # Compass title and current direction
        self.canvas.create_text(
            cx, cy - radius - 35,
            text="LISTENING",
            fill=self.hex_with_alpha('#00d4ff', 0.9),
            font=('Arial', 10, 'bold'),
            tags='compass'
        )
        
        self.canvas.create_text(
            cx, cy - radius - 50,
            text="COMPASS",
            fill=self.hex_with_alpha('#ff00ff', 0.8),
            font=('Arial', 8, 'bold'),
            tags='compass'
        )
        
        # Enhanced current direction display
        self.canvas.create_text(
            cx, cy + radius + 15,
            text=f"Facing: {direction}",
            fill='white',
            font=('Arial', 10, 'bold'),
            tags='compass'
        )
        
        self.canvas.create_text(
            cx, cy + radius + 30,
            text=f"{azimuth:.0f}°",
            fill='#00d4ff',
            font=('Arial', 9, 'bold'),
            tags='compass'
        )
        
        # Removed the blue listening beam - pink waves on listener sprite are enough
    
        

    def get_source_color(self, name):
        """Get intelligent color coding based on source name (matching web player)."""
        name_lower = name.lower()
        if 'rain' in name_lower: 
            return '#4080ff'  # Blue for rain
        elif 'bass' in name_lower: 
            return '#ff40ff'  # Magenta for bass
        elif 'kick' in name_lower: 
            return '#ff4040'  # Red for kick
        elif 'melody' in name_lower: 
            return '#40ff40'  # Green for melody
        elif 'pad' in name_lower: 
            return '#ffff40'  # Yellow for pad
        else: 
            return '#00d4ff'  # Default cyan

    def draw_sources_simple(self, cx, cy):
        """Draw sources with web player styling - 15px radius, gradient glows, intelligent colors."""
        
        for i, (name, source) in enumerate(self.sources.items()):
            # Match web player visualization:
            # Screen X = World X (left/right)
            # Screen Y = -World Z (forward is UP, backward is DOWN)
            # World Y = elevation (visual effects)
            x = cx + source['position'][0] * self.scale  # X: left/right
            y = cy - source['position'][2] * self.scale  # Z: forward(+) is UP on screen
            elevation = source['position'][1]  # Y: elevation (height)
            
            # Intelligent color coding like web player
            color = self.get_source_color(name)
            
            # Web player style - 15px radius with subtle pulse
            radius = 15 + math.sin(self.time * 0.8 + len(name)) * 1
            
            # Gradient glow effect (web player style)
            glow_steps = 3
            for j in range(glow_steps):
                glow_radius = radius + (j + 1) * 5
                alpha = (0.4 - j * 0.1)
                glow_color = self.hex_with_alpha(color, alpha)
                self.canvas.create_oval(
                    x - glow_radius, y - glow_radius,
                    x + glow_radius, y + glow_radius,
                    fill=glow_color, outline='', tags='source_glow'
                )
            
            # Core circle (web player style)
            self.canvas.create_oval(
                x - radius, y - radius, x + radius, y + radius,
                fill=color, outline='', tags='source'
            )
            
            # Label
            self.canvas.create_text(
                x, y - radius - 8,
                text=name, fill='white', font=('Arial', 10), tags='source'
            )
            
            # Elevation display - simple dotted line (web player style)
            if abs(elevation) > 0.1:
                elev_color = self.hex_with_alpha(color, 0.6)
                line_length = elevation * self.scale * 0.5
                
                # Dotted line from source center
                num_dashes = max(3, int(abs(line_length) / 5))
                for dash_i in range(num_dashes):
                    dash_start_y = y + line_length * dash_i / num_dashes
                    dash_end_y = y + line_length * (dash_i + 0.6) / num_dashes
                    self.canvas.create_line(
                        x, dash_start_y, x, dash_end_y,
                        fill=elev_color, width=2, tags='source_elevation'
                    )

    def update_source_position(self, source_id, position):
        """Update a source position for real-time positioning feedback."""
        # Find source by ID and update position
        for source_name, source_data in self.sources.items():
            if source_data.get('id') == source_id:
                self.sources[source_name]['position'] = position
                
                # Update spatial engine if it's running
                if hasattr(self.spatial_engine, 'update_source_position'):
                    self.spatial_engine.update_source_position(source_id, position)
                    
                logger.debug(f"Player: Updated {source_name} position to {position}")
                break
                
    def update_source_positions_from_engine(self):
        """Sync source positions from main audio engine."""
        if not hasattr(self.parent_window, 'audio_engine'):
            return
            
        for source_id, source in self.parent_window.audio_engine.sources.items():
            self.update_source_position(source_id, source.position)
    
    def enable_preview_mode(self):
        """Enable real-time preview mode for positioning."""
        if not self.is_playing:
            # Start audio preview for positioning
            self.toggle_playback()
            logger.info("Preview mode: Audio started for positioning")
    
    def disable_preview_mode(self):
        """Disable preview mode."""
        if self.is_playing:
            self.stop_playback()
            logger.info("Preview mode: Audio stopped")
    
    def _ensure_pygame_loaded(self):
        """Lazily load pygame when needed."""
        global PYGAME_AVAILABLE, pygame
        
        if PYGAME_AVAILABLE is not None:
            return PYGAME_AVAILABLE
            
        try:
            import pygame as pg
            pygame = pg
            PYGAME_AVAILABLE = True
            return True
        except ImportError:
            PYGAME_AVAILABLE = False
            logger.warning("pygame not available - gamepad support disabled")
            return False
    
    def _setup_gamepad(self):
        """Initialize gamepad support."""
        # Only try to load pygame when gamepad setup is actually called
        if not self._ensure_pygame_loaded():
            return
            
        try:
            pygame.init()
            pygame.joystick.init()
            logger.info(f"Gamepad system initialized. Controllers detected: {pygame.joystick.get_count()}")
            
            # Check for existing controllers
            if pygame.joystick.get_count() > 0:
                self.gamepad = pygame.joystick.Joystick(0)
                self.gamepad.init()
                self.gamepad_connected = True
                self._show_gamepad_notification(f"Controller connected: {self.gamepad.get_name()}")
                logger.info(f"Gamepad connected: {self.gamepad.get_name()}")
                
            # Start polling loop
            self._poll_gamepad()
            
        except Exception as e:
            logger.warning(f"Gamepad initialization failed: {e}")
            
    def _poll_gamepad(self):
        """Poll gamepad state (matches web player polling)."""
        if not self._ensure_pygame_loaded():
            return
            
        try:
            # Process pygame events to detect connect/disconnect
            pygame.event.pump()
            
            # Check for controller connection changes
            current_count = pygame.joystick.get_count()
            
            if current_count > 0 and not self.gamepad_connected:
                # Controller connected
                self.gamepad = pygame.joystick.Joystick(0)
                self.gamepad.init()
                self.gamepad_connected = True
                self._show_gamepad_notification(f"Controller connected: {self.gamepad.get_name()}")
                logger.info(f"Gamepad connected: {self.gamepad.get_name()}")
                
            elif current_count == 0 and self.gamepad_connected:
                # Controller disconnected
                self.gamepad_connected = False
                self.gamepad = None
                self._show_gamepad_notification("Controller disconnected")
                logger.info("Controller disconnected")
                
            # Poll controller input if connected
            if self.gamepad_connected and self.gamepad:
                self._handle_gamepad_input()
                
        except Exception as e:
            logger.warning(f"Gamepad polling error: {e}")
            
        # Schedule next poll (60fps like web player)
        if hasattr(self, 'canvas'):
            self.canvas.after(16, self._poll_gamepad)
            
    def _handle_gamepad_input(self):
        """Handle gamepad input (matches web player controls exactly)."""
        if not self.gamepad:
            return
            
        try:
            # Left stick for movement (axes 0 and 1)
            left_stick_x = self.gamepad.get_axis(0)  # X axis
            left_stick_y = self.gamepad.get_axis(1)  # Y axis
            
            # Right stick for rotation (axes 2 and 3) 
            right_stick_x = self.gamepad.get_axis(2) if self.gamepad.get_numaxes() > 2 else 0
            right_stick_y = self.gamepad.get_axis(3) if self.gamepad.get_numaxes() > 3 else 0
            
            # Apply deadzone (same as web player)
            deadzone = 0.15
            
            # Movement (absolute world coordinates - not rotated by listening direction)
            if abs(left_stick_x) > deadzone or abs(left_stick_y) > deadzone:
                # Absolute movement: stick right = east, stick up = north
                dx = left_stick_x * 0.33    # Right = positive X (east)
                dy = 0                      # No vertical movement on left stick  
                dz = -left_stick_y * 0.33   # Up on stick = positive Z (north)
                
                # Update position directly - NO rotation applied
                self.listener_pos['x'] += dx  # Always east/west
                self.listener_pos['z'] += dz  # Always north/south
                
                # Update spatial engine
                self.spatial_engine.update_listener_position(
                    self.listener_pos['x'],
                    self.listener_pos['y'],
                    self.listener_pos['z']
                )
                
                # Request render for movement
                self.request_render()
                
            # Right stick for absolute direction (matches web player)
            if abs(right_stick_x) > deadzone or abs(right_stick_y) > deadzone:
                # Convert stick position to absolute angle (same as web player)
                target_angle = math.atan2(-right_stick_x, -right_stick_y) * 180 / math.pi
                self.listener_rot['azimuth'] = target_angle
                
                # Update spatial engine
                self.spatial_engine.update_listener_rotation(
                    self.listener_rot['azimuth'],
                    self.listener_rot['elevation']
                )
                
                self.request_render()
                
            # Button controls (matches web player button mapping)
            num_buttons = self.gamepad.get_numbuttons()
            
            # A button (0) - Play/Pause
            if num_buttons > 0:
                button_0_pressed = self.gamepad.get_button(0)
                if button_0_pressed and not self.gamepad_button_states.get(0, False):
                    self.toggle_playback()
                self.gamepad_button_states[0] = button_0_pressed
                
            # Y button (3) - Reset position  
            if num_buttons > 3:
                button_3_pressed = self.gamepad.get_button(3)
                if button_3_pressed and not self.gamepad_button_states.get(3, False):
                    self.reset_position()
                self.gamepad_button_states[3] = button_3_pressed
                
            # Bumpers for elevation (matches web player)
            # Right bumper (5) - Move up
            if num_buttons > 5 and self.gamepad.get_button(5):
                self.listener_pos['y'] += 0.02  # Slower elevation movement
                self.spatial_engine.update_listener_position(
                    self.listener_pos['x'],
                    self.listener_pos['y'], 
                    self.listener_pos['z']
                )
                self.request_render()
                
            # Left bumper (4) - Move down
            if num_buttons > 4 and self.gamepad.get_button(4):
                self.listener_pos['y'] -= 0.02  # Slower elevation movement
                self.spatial_engine.update_listener_position(
                    self.listener_pos['x'],
                    self.listener_pos['y'],
                    self.listener_pos['z']
                )
                self.request_render()
                
        except Exception as e:
            logger.warning(f"Gamepad input error: {e}")
            
    def _show_gamepad_notification(self, message):
        """Show gamepad notification (matches web player notifications)."""
        if self.gamepad_notification_label:
            self.gamepad_notification_label.config(text=message)
            
            # Update controls display when gamepad status changes
            self.update_controls_display()
            
            # Hide after 3 seconds (same as web player)
            def hide_notification():
                if self.gamepad_notification_label:
                    self.gamepad_notification_label.config(text="")
                    
            if hasattr(self, 'canvas'):
                self.canvas.after(3000, hide_notification)
                
    def update_controls_display(self):
        """Update controls display based on gamepad connection status."""
        if not hasattr(self, 'controls_label'):
            return
            
        if self.gamepad_connected:
            # Show gamepad controls when connected
            controls_text = "🎮 Left Stick: Move | Right Stick: Look | A: Play/Pause | Y: Reset | Bumpers: Elevation | " \
                          "⌨️ WASD: Move | Arrows: Look | Space: Play/Pause | Q/E: Up/Down | R: Reset"
        else:
            # Show keyboard controls when no gamepad
            controls_text = "⌨️ WASD: Move | Arrow Keys: Look | Q/E: Up/Down | Space: Play/Pause | R: Reset | Ctrl+Scroll: Zoom"
            
        self.controls_label.config(text=controls_text)

    def cleanup(self):
        """Clean up resources."""
        self.animation_running = False
        
        # Cleanup gamepad
        if PYGAME_AVAILABLE and self.gamepad_connected and pygame:
            try:
                if self.gamepad:
                    self.gamepad.quit()
                pygame.joystick.quit()
                pygame.quit()
            except Exception as e:
                logging.warning(f"Failed to cleanup pygame/gamepad: {e}")
                
        if self.spatial_engine:
            self.spatial_engine.cleanup()