"""
Theme Helper Functions
Quick utilities for applying consistent theming across panels
"""

import tkinter as tk
from tkinter import ttk
from . import theme


def create_themed_button(parent, text, command, icon=None, style='normal', **kwargs):
    """
    Create a professionally themed button

    Args:
        parent: Parent widget
        text: Button text
        command: Button command
        icon: Optional PhotoImage icon
        style: 'normal', 'accent', 'success', or 'danger'
        **kwargs: Additional button config options

    Returns:
        Configured tk.Button
    """
    # Determine colors based on style
    if style == 'accent':
        bg = theme.ACCENT_PRIMARY
        hover_bg = theme.ACCENT_HOVER
        fg = '#FFFFFF'
    elif style == 'success':
        bg = theme.SUCCESS
        hover_bg = '#45a049'
        fg = '#FFFFFF'
    elif style == 'danger':
        bg = theme.ERROR
        hover_bg = '#d32f2f'
        fg = '#FFFFFF'
    else:  # normal
        bg = theme.BUTTON_NORMAL
        hover_bg = theme.BUTTON_HOVER
        fg = theme.TEXT_PRIMARY

    # Create button
    btn = tk.Button(
        parent,
        text=text,
        command=command,
        bg=bg,
        fg=fg,
        activebackground=hover_bg,
        activeforeground=fg if style != 'normal' else theme.TEXT_PRIMARY,
        relief='flat',
        borderwidth=0,
        padx=kwargs.get('padx', 12),
        pady=kwargs.get('pady', 8),
        font=kwargs.get('font', theme.FONT_NORMAL),
        cursor='hand2',
        **{k: v for k, v in kwargs.items() if k not in ['padx', 'pady', 'font']}
    )

    if icon:
        btn.config(image=icon, compound='left')

    # Add hover effect
    def on_enter(e):
        btn.config(bg=hover_bg)

    def on_leave(e):
        btn.config(bg=bg)

    btn.bind('<Enter>', on_enter)
    btn.bind('<Leave>', on_leave)

    return btn


def create_themed_label(parent, text, style='normal', **kwargs):
    """
    Create a professionally themed label

    Args:
        parent: Parent widget
        text: Label text
        style: 'normal', 'secondary', 'title', 'small'
        **kwargs: Additional label config options

    Returns:
        Configured tk.Label
    """
    if style == 'title':
        font = theme.FONT_TITLE
        fg = theme.TEXT_PRIMARY
    elif style == 'secondary':
        font = theme.FONT_NORMAL
        fg = theme.TEXT_SECONDARY
    elif style == 'small':
        font = theme.FONT_SMALL
        fg = theme.TEXT_SECONDARY
    else:  # normal
        font = theme.FONT_NORMAL
        fg = theme.TEXT_PRIMARY

    return tk.Label(
        parent,
        text=text,
        font=kwargs.get('font', font),
        fg=kwargs.get('fg', fg),
        bg=kwargs.get('bg', theme.BACKGROUND_SECONDARY),
        **{k: v for k, v in kwargs.items() if k not in ['font', 'fg', 'bg']}
    )


def create_themed_frame(parent, **kwargs):
    """Create a themed frame"""
    return tk.Frame(
        parent,
        bg=kwargs.get('bg', theme.BACKGROUND_SECONDARY),
        **{k: v for k, v in kwargs.items() if k != 'bg'}
    )


def configure_ttk_treeview_theme(style_name='Themed.Treeview'):
    """
    Configure a ttk.Style for themed treeview

    Args:
        style_name: Name for the style

    Returns:
        style_name to use in Treeview widget
    """
    style = ttk.Style()
    style.configure(
        style_name,
        background=theme.BACKGROUND_INPUT,
        fieldbackground=theme.BACKGROUND_INPUT,
        foreground=theme.TEXT_PRIMARY,
        borderwidth=0
    )
    style.configure(
        f'{style_name}.Heading',
        background=theme.BACKGROUND_TERTIARY,
        foreground=theme.TEXT_PRIMARY,
        borderwidth=0
    )
    style.map(style_name,
              background=[('selected', theme.ACCENT_PRIMARY)],
              foreground=[('selected', '#FFFFFF')])

    return style_name


def configure_ttk_scale_theme(style_name='Themed.Horizontal.TScale'):
    """
    Configure a ttk.Style for themed slider/scale

    Args:
        style_name: Name for the style

    Returns:
        style_name to use in Scale widget
    """
    style = ttk.Style()
    style.configure(
        style_name,
        background=theme.BACKGROUND_SECONDARY,
        troughcolor=theme.SLIDER_TRACK,
        borderwidth=0,
        lightcolor=theme.ACCENT_PRIMARY,
        darkcolor=theme.ACCENT_PRIMARY,
        sliderthickness=16
    )

    return style_name
